(function($) {
  // ✅ フォールバック辞書を取得  
  // ✅ Retrieve fallback dictionary
  const fallback = window['sbp2_i18n_fallback_sbp2_status'] || {};

  // ✅ ハイブリッド翻訳関数：__() → fallback の順に参照  
  // ✅ Hybrid translation function: Refer to __() then fallback
  function T__(text) {
    if (typeof wp !== 'undefined' && typeof wp.i18n?.__ === 'function') {
      const result = wp.i18n.__(text, 'smart-booking-pro2'); // ✅ 正しい呼び出し
      if (result !== text) return result;
    }
    return fallback[text]?.[1] || text;
  }

// プロ版購入ボタンクリック処理  
// Process for Pro version purchase button click
jQuery(document).ready(function ($) {
  $('.sbp2-checkout-btn').on('click', async function (e) {
    e.preventDefault();

    // 🔐 認証DBから秘密キーを取得  
    // 🔐 Retrieve secret key from authentication DB
    let jwtSecret = '';
    try {
      jwtSecret = await fetchJwtSecretFromAuthDB();
    } catch (err) {
      alert(T__('認証に失敗しました：') + err.message);
      return;
    }

    const priceId = $(this).data('price-id');
    if (!priceId) {
      alert(T__('価格IDが見つかりません。'));
      return;
    }

    const payload = {
      uuid: SBP2.uuid || '',
      site_url: SBP2.site_url || window.location.origin,
      admin_email: SBP2.admin_email || '',
      wp_version: SBP2.wp_version || '',
      php_version: SBP2.php_version || '',
      plugin_version: SBP2.plugin_version || '',
      installed_at: SBP2.installed_at || '',
      installed_by_email: SBP2.installed_by_email || '',
      installed_by_user: SBP2.installed_by_user || '',
      theme: SBP2.theme || '',
      price_id: priceId,
      plan_type: priceId.replace('price_', ''),
      referer: SBP2.referer || '',
      timezone: Intl.DateTimeFormat().resolvedOptions().timeZone || '',
      locale: navigator.language || 'ja'
    };

    if (!payload.uuid || !payload.admin_email) {
      alert(T__('UUIDかメールアドレスが不足しています。'));
      return;
    }

    function base64UrlEncode(obj) {
      return btoa(JSON.stringify(obj))
        .replace(/=/g, '')
        .replace(/\+/g, '-')
        .replace(/\//g, '_');
    }

    const header = { alg: 'HS256', typ: 'JWT' };
    const iat = Math.floor(Date.now() / 1000);
    const exp = iat + 300;

    const jwtPayload = {
      uuid: payload.uuid,
      referer: payload.referer,
      iat: iat,
      exp: exp
    };

    const encodedHeader = base64UrlEncode(header);
    const encodedPayload = base64UrlEncode(jwtPayload);

    const signature = CryptoJS.HmacSHA256(encodedHeader + '.' + encodedPayload, jwtSecret);
    const encodedSignature = CryptoJS.enc.Base64.stringify(signature)
      .replace(/=/g, '')
      .replace(/\+/g, '-')
      .replace(/\//g, '_');

    const jwt = encodedHeader + '.' + encodedPayload + '.' + encodedSignature;

    // 🧾 決済セッション作成リクエスト  
    // 🧾 Payment session creation request
    fetch('https://arce-experience.com/wp-json/sbp2/v1/create-checkout-session', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': 'Bearer ' + jwt
      },
      body: JSON.stringify(payload)
    })
      .then(res => res.json())
      .then(async (data) => {
        if (data.success && data.url) {
          window.location.href = data.url;
        } else {
          // ❗ ステータス同期処理を先に実行
          try {
            await fetch(ajaxurl, {
              method: 'POST',
              headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
              body: new URLSearchParams({ action: 'sbp2_sync_status_from_auth_db' })
            });
          } catch (syncErr) {
            console.warn(T__('状態同期に失敗:'), syncErr);
          }

          // ⚠️ アラート出力
          alert(data.message || T__('決済リンクの生成に失敗しました。'));
        }
      })
      .catch(err => {
        console.error('通信エラー:', err);
        alert(T__('通信エラーが発生しました。'));
      });
  });
});



// プラン変更ボタンクリック処理  
// Process for plan change button click
document.addEventListener('DOMContentLoaded', function () {
  const button = document.getElementById('sbp2-change-plan-button');

  if (button) {
    button.addEventListener('click', async function (e) {
      e.preventDefault();
      button.disabled = true;
      button.textContent = T__('認証中...');

      const uuid = SBP2.uuid;
      const referer = SBP2.referer;

      try {

        const jwtSecret = await fetchJwtSecretFromAuthDB();

        const { customerId, customerEmail } = await fetchStripeInfoFromWP();

        if (!customerId) throw new Error(T__('Customer IDが見つかりません。'));

        // 🔏 JWT生成  
        // 🔏 Generate JWT
        function base64UrlEncode(obj) {
          return btoa(JSON.stringify(obj))
            .replace(/=/g, '')
            .replace(/\+/g, '-')
            .replace(/\//g, '_');
        }

        const header = { alg: 'HS256', typ: 'JWT' };
        const iat = Math.floor(Date.now() / 1000);
        const exp = iat + 300;
        const issued_at_human = new Date(iat * 1000).toISOString(); // UTCのISO形式


        const payload = {
          uuid: uuid,
          referer: referer,
          customer_id: customerId,
          email: customerEmail,
          action: 'change_plan',
          iat: iat,
          exp: exp
        };

        const encodedHeader = base64UrlEncode(header);
        const encodedPayload = base64UrlEncode(payload);

        const signature = CryptoJS.HmacSHA256(
          encodedHeader + '.' + encodedPayload,
          jwtSecret
        );
        const encodedSignature = CryptoJS.enc.Base64.stringify(signature)
          .replace(/=/g, '')
          .replace(/\+/g, '-')
          .replace(/\//g, '_');

        const jwt = encodedHeader + '.' + encodedPayload + '.' + encodedSignature;

        // 🎯 認証DBへポータル生成リクエスト  
        // 🎯 Portal generation request to authentication DB
        const res = await fetch(SBP2.portal_url, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': 'Bearer ' + jwt
          },
          body: JSON.stringify({
            uuid: uuid,
            referer: referer,
            customer_id: customerId,
            email: customerEmail,
            action: 'change_plan',
            issued_at_human: issued_at_human,
            locale: navigator.language || 'ja'
          })
        });

        const result = await res.json();

        if (result.success && result.portal_url) {
          window.location.href = result.portal_url;
        } else {
          throw new Error(result.message || T__('ポータルリンクの取得に失敗しました'));
        }

      } catch (err) {
        alert(T__('エラーが発生しました：') + err.message);
        button.textContent = T__('プランを変更・確認する');
      } finally {
        button.disabled = false;
      }
    });
  }
});


//************************************************************************************************************************* */
// 認証関数とID・メールアドレス取得
//************************************************************************************************************************* */

/**
 * 🔐 認証DBに UUID + referer を送信して秘密キーを取得し、WordPress側に保存する関数  
 * 🔐 Function to send UUID + referer to authentication DB, obtain secret key, and save it on WordPress side
 */
async function fetchJwtSecretFromAuthDB() {
  const uuid = SBP2.uuid || '';
  const referer = SBP2.referer || '';
  const locale  = navigator.language || 'ja';

  if (!uuid || !referer) {
    throw new Error(T__('UUID または referer が指定されていません。'));
  }

  try {
    // ✅ 認証DBからJWT秘密キー取得  
    // ✅ Obtain JWT secret key from authentication DB
    const response = await fetch('https://arce-experience.com/wp-json/sbp2/v1/authenticate', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ uuid, referer, locale })
    });

    const result = await response.json();

    if (response.ok && result.success && result.jwt_secret) {
      const jwtSecret = result.jwt_secret;

      const saveResponse = await fetch(SBP2.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
          action: 'sbp2_save_jwt_secret',
          jwt_secret: jwtSecret
        })
      });

      const saveResult = await saveResponse.json();
      if (!saveResult.success) {
        throw new Error(saveResult.message || T__('秘密キーの保存に失敗しました。'));
      }

      return jwtSecret;
    } else {
      throw new Error(result.message || T__('認証に失敗しました。'));
    }
  } catch (error) {
    console.error('[SBP2] 認証通信エラー:', error);
    throw error;
  }
}

async function fetchStripeInfoFromWP() {
  try {
    const res = await fetch(SBP2.ajaxurl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: new URLSearchParams({ action: 'sbp2_get_stripe_info' })
    });

    const json = await res.json();
    if (!json.success) {
      throw new Error(json.message || T__('取得失敗'));
    }

    return {
      customerId: json.data.customer_id,
      customerEmail: json.data.email || ''
    };
  } catch (err) {
    console.error('[SBP2] Stripe情報の取得エラー:', err);
    throw err;
  }
}

document.addEventListener('DOMContentLoaded', function () {
  const status = window.SBP2_SUBSCRIPTION_STATUS || 'inactive';
  if (status !== 'inactive') {
    document.querySelectorAll('.sbp2-checkout-btn').forEach(btn => {
      btn.classList.add('disabled');
      btn.setAttribute('aria-disabled', 'true');
      btn.style.pointerEvents = 'none';
      btn.style.opacity = '0.5';
      btn.addEventListener('click', function (e) {
        e.preventDefault();
        alert(T__('現在のステータスではこの操作はできません。'));
      });
    });
  }
});

})(jQuery);