<?php
// CF7 ショートコードを生成する関数  
// Function to generate CF7 shortcode
if (!defined('ABSPATH')) {
    exit;
}
function cf7_generate_shortcode($type = 'text', $name = '', $required = false, $hidden = false, $class = '', $id = '', $placeholder = '', $default_value = '', $use_label = false, $options = []) {
    if (empty($name)) {
        $name = 'textname-' . date('His');
    }

    $required_mark = $required ? '*' : '';
    $shortcode = "[$type$required_mark $name";

    if ($hidden) {
        $shortcode = "[hidden $name";
    }

    if (!empty($class)) $shortcode .= " class:$class";
    if (!empty($id)) $shortcode .= " id:$id";
    if (!empty($placeholder)) $shortcode .= ' placeholder "' . esc_attr($placeholder) . '"';
    if (!empty($default_value)) $shortcode .= ' "' . esc_attr($default_value) . '"';

    if (!empty($options) && in_array($type, ['checkbox', 'radio', 'select'])) {
        if ($use_label) {
            $shortcode .= ' use_label_element';
        }
        foreach ($options as $option) {
            $shortcode .= ' "' . esc_attr($option) . '"';
        }
    }

    $shortcode .= ']';
    return $shortcode;
}

// CF7のショートコードを解析し、ACFフィールドに変換する関数  
// Function to parse CF7 shortcode and convert to ACF fields
function cf7_parse_shortcode($shortcode) {
    $pattern = '/\[(\w+)(\*?)(?:\s([\w-]+))?(.*?)]/';
    preg_match($pattern, $shortcode, $matches);

    if (!$matches) return false;

    $acf_field = [
        'name' => $matches[3] ?? 'field-' . date('His'),
        'type' => 'text',
        'required' => !empty($matches[2]),
        'class' => '',
        'id' => '',
        'placeholder' => '',
        'default' => '',
        'options' => [],
        'use_label' => false
    ];

    $cf7_to_acf = [
        'text' => 'text',
        'email' => 'email',
        'tel' => 'text',
        'textarea' => 'textarea',
        'radio' => 'radio',
        'checkbox' => 'checkbox',
        'select' => 'select',
        'date' => 'date_picker',
        'number' => 'number'
    ];

    $acf_field['type'] = $cf7_to_acf[$matches[1]] ?? 'text';

    if (!empty($matches[4])) {
        preg_match_all('/(\w+):([\w-]+)/', $matches[4], $attr_matches, PREG_SET_ORDER);
        foreach ($attr_matches as $attr) {
            switch ($attr[1]) {
                case 'class': $acf_field['class'] = $attr[2]; break;
                case 'id': $acf_field['id'] = $attr[2]; break;
            }
        }
        preg_match('/placeholder\s+"([^"]+)"/', $matches[4], $placeholder);
        if (!empty($placeholder[1])) {
            $acf_field['placeholder'] = $placeholder[1];
        }
    }

    return $acf_field;
}

function smart_booking_create_cf7_form() {
    check_ajax_referer('smart_booking_nonce', 'nonce');

    if (!class_exists('WPCF7_ContactForm')) {
        wp_send_json_error(['message' => esc_html__('Contact Form 7 がインストールされていません。', 'smart-booking-pro2')]);
    }

    // ✅ **AJAXで受け取ったデータを取得**  
    // ✅ **Retrieve data received via AJAX**
    $form_name = sanitize_text_field($_POST['form_name']);
    $admin_email = sanitize_email($_POST['admin_email']);
    $from_email = sanitize_email($_POST['from_email']);
    $email_subject = sanitize_text_field($_POST['email_subject']);
    $additional_headers = sanitize_textarea_field($_POST['additional_headers']);
    $admin_mail_intro = sanitize_textarea_field($_POST['admin_mail_intro']); // ✅ 追加
    $admin_mail_closing = sanitize_textarea_field($_POST['admin_mail_closing']); // ✅ 追加
    $fields = $_POST['fields'] ?? [];

    if (empty($form_name) || empty($fields)) {
        wp_send_json_error(['message' => esc_html__('フォーム名とフィールドを入力してください。', 'smart-booking-pro2')]);
    }

    // ✅ **フォームのHTMLを生成**  
    // ✅ **Generate form HTML**
    $form_content = "";
    foreach ($fields as $field) {
        if (empty($field['type'])) {
            error_log("⚠️ フィールド「{$field['name']}」のタイプが未設定のためスキップ");
            continue;
        }

        $field_name = sanitize_text_field($field['name']);
        $field_label = sanitize_text_field($field['label']);
        $field_type = sanitize_text_field($field['type']);
        $required = !empty($field['required']) && $field['required'] == "true" ? '*' : ''; 
        $class_attr = !empty($field['className']) ? " class:{$field['className']}" : "";
        $id_attr = !empty($field['idName']) ? " id:{$field['idName']}" : "";
        $placeholder = !empty($field['placeholder']) ? ' placeholder "' . esc_attr($field['placeholder']) . '"' : "";

        $default_value = !empty($field['defaultValue']) ? ' "' . esc_attr($field['defaultValue']) . '"' : "";

        $raw_options = isset($field['options']) && is_string($field['options']) ? sanitize_text_field($field['options']) : '';
        $options = array_map('trim', explode(',', $raw_options));
        $wrap_label = !empty($field['wraplabel']) && $field['wraplabel'] == "true" ? " use_label_element" : ""; 

        // ✅ **Hidden フィールドの処理**
        if (!empty($field['hidden']) && $field['hidden'] == "true") {
            $form_content .= "[hidden {$field_name}{$default_value}]\n\n";
            error_log("✅ [hidden {$field_name}{$default_value}] を追加");
            continue;
        }

        // ✅ **ラベルの追加**  
        // ✅ **Add label**
        if (!empty($field_label)) {
            $form_content .= "<label>{$field_label}</label>\n";
        }

        // ✅ **フィールドのタイプごとに処理**  
        // ✅ **Process by field type**
        switch ($field_type) {
            case 'text':
            case 'email':
            case 'tel':
            case 'number':
            case 'textarea':
            case 'date':
                $form_content .= "[{$field_type}{$required} {$field_name}{$class_attr}{$id_attr}{$placeholder}{$default_value}]\n\n";
                break;

            case 'checkbox':
            case 'radio':
            case 'select':
                $formatted_options = array_map(fn($opt) => "\"{$opt}\"", $options);
                $form_content .= "[{$field_type}{$required} {$field_name}{$class_attr}{$id_attr}{$wrap_label} " . implode(" ", $formatted_options) . "{$default_value}]\n\n";
                break;

            default:
                error_log("⚠️ 未知のフィールドタイプ: " . $field_type);
                break;
        }
    }

    $form_content .= "[submit '" . esc_html__('送信', 'smart-booking-pro2') . "']";


    // ✅ **フォームをWPに登録**  
    // ✅ **Register form to WP**
    $post_data = [
        'post_title'  => $form_name,
        'post_status' => 'publish',
        'post_type'   => 'wpcf7_contact_form',
    ];

    $form_id = wp_insert_post($post_data);
    if (is_wp_error($form_id) || !$form_id) {
        wp_send_json_error(['message' => esc_html__('フォームの作成に失敗しました。', 'smart-booking-pro2')]);
    }
    

    update_post_meta($form_id, '_form', $form_content);

    // ✅ **メール設定**  
    // ✅ **Email settings**
    $admin_mail_body = $admin_mail_intro . "\n\n"; // ✅ 追加
    foreach ($fields as $field) {
        $admin_mail_body .= "{$field['label']}: [{$field['name']}]\n";
    }
    $admin_mail_body .= "\n\n" . $admin_mail_closing; // ✅ 追加

    $admin_mail_template = [
        'active'             => true,
        'recipient'          => $admin_email,
        'sender'             => $from_email,
        'subject'            => $email_subject,
        'body'               => $admin_mail_body,
        'additional_headers' => $additional_headers,
        'attachments'        => '',
        'use_html'           => false,
    ];

    // ✅ **CF7 APIでメール設定を適用**  
    // ✅ **Apply email settings via CF7 API**
    $contact_form = WPCF7_ContactForm::get_instance($form_id);
    if ($contact_form) {
        $contact_form->set_properties([
            'form' => $form_content,
            'mail' => $admin_mail_template,
        ]);
        $contact_form->save();
    }

    // ✅ **成功レスポンス**  
    // ✅ **Success response**
    wp_send_json_success([
    'message' => esc_html__('フォームが作成されました！', 'smart-booking-pro2'),
        'form_id' => $form_id,
        'shortcode' => '[contact-form-7 id="' . $form_id . '" title="' . esc_attr($form_name) . '"]',
    ]);
}
add_action('wp_ajax_smart_booking_create_cf7_form', 'smart_booking_create_cf7_form');
