<?php
if (!defined('ABSPATH')) {
    exit;
}

// 予約データの取得処理  
// Reservation data retrieval process
function smart_booking_get_post_data() {

    $post_type = sanitize_text_field($_POST['post_type']);
    $start_date = !empty($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : '';
    $end_date = !empty($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : '';

    $query_args = array(
        'post_type'      => $post_type,
        'posts_per_page' => -1,
        'post_status'    => 'any',
    );

    // ✅ 予約済み日時（reserved-daytime）の場合、カスタムフィールド「予約日（reserved-day）」でフィルタリング  
    // ✅ For reserved-daytime, filter by custom field "reserved-day"
    if ($post_type === 'reserved-daytime') {
        $meta_query = array('relation' => 'AND');

        if (!empty($start_date)) {
            $meta_query[] = array(
                'key'     => 'reserved-day',
                'value'   => $start_date,
                'compare' => '>=',
                'type'    => 'DATE'
            );
        }

        if (!empty($end_date)) {
            $meta_query[] = array(
                'key'     => 'reserved-day',
                'value'   => $end_date,
                'compare' => '<=',
                'type'    => 'DATE'
            );
        }

        $query_args['meta_query'] = $meta_query;
        $query_args['post_status'] = 'any';
    } else {
        // ✅ 他の投稿タイプの場合、登録日でフィルタリング  
        // ✅ For other post types, filter by registration date
        if (!empty($start_date) || !empty($end_date)) {
            $query_args['date_query'] = array();

            if (!empty($start_date)) {
                $query_args['date_query'][] = array(
                    'after' => $start_date,
                    'inclusive' => true
                );
            }

            if (!empty($end_date)) {
                $query_args['date_query'][] = array(
                    'before' => $end_date,
                    'inclusive' => true
                );
            }
        }
    }

    $query = new WP_Query($query_args);

    if ($query->have_posts()) {
        ob_start();
        echo '<table class="widefat"><thead><tr>';
        echo '<th>' . esc_html__('ID', 'smart-booking-pro2') . '</th>';
        echo '<th>' . esc_html__('タイトル', 'smart-booking-pro2') . '</th>';
        echo '<th>' . esc_html__('登録日', 'smart-booking-pro2') . '</th>';
        echo '</tr></thead><tbody>';

        while ($query->have_posts()) {
            $query->the_post();
            echo '<tr>';
            echo '<td>' . get_the_ID() . '</td>';
            echo '<td>' . esc_html(get_the_title()) . '</td>';
            echo '<td>' . get_the_date('Y-m-d') . '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        $html = ob_get_clean();

        wp_send_json_success(['html' => $html]);
    } else {
        wp_send_json_error([
            'message' => esc_html__('データが見つかりません。', 'smart-booking-pro2')
        ]);
    }


    wp_die();
}
add_action('wp_ajax_smart_booking_get_post_data', 'smart_booking_get_post_data');



function smart_booking_get_acf_fields() {
    // ✅ セキュリティチェック
    check_ajax_referer('smart_booking_nonce', 'nonce');

    if (!isset($_POST['post_type'])) {
        wp_send_json_error([
            'message' => esc_html__('投稿タイプが指定されていません。', 'smart-booking-pro2')
        ]);
    }

    $post_type = sanitize_text_field($_POST['post_type']);

    // ✅ 保存されたACF構成情報を取得
    $saved_acf_data = get_option('smart_booking_acf_data', []);
    $saved_acf_db   = get_option('smart_booking_acf_db', []);
    $merged_data    = array_merge($saved_acf_data, $saved_acf_db);

    $acf_fields = [];
    foreach ($merged_data as $data) {
        if ($data['post_type'] === $post_type) {
            foreach ($data['acf_fields'] as $field) {
                if (is_array($field) && isset($field['name'], $field['type'])) {
                    $acf_fields[] = [
                        'name'  => $field['name'],
                        'type'  => $field['type'],
                        'label' => $field['label'] ?? '', // 任意
                    ];
                } elseif (is_array($field) && isset($field[0], $field[1])) {
                    // 念のため古い形式も対応
                    $acf_fields[] = [
                        'name'  => $field[0],
                        'type'  => $field[1],
                        'label' => $field[2] ?? '',
                    ];
                }
            }
        }
    }

    if (empty($acf_fields)) {
        wp_send_json_error([
            'message' => esc_html__('ACFフィールドが見つかりません。', 'smart-booking-pro2')
        ]);
    }

    wp_send_json_success(['fields' => $acf_fields]);
}

add_action('wp_ajax_smart_booking_get_acf_fields', 'smart_booking_get_acf_fields');



function save_reservation_data() {
    // 🔐 nonce 検証（エラー時は即終了）  
    // 🔐 Nonce verification (exit immediately on error)
    if (!isset($_POST['_ajax_nonce']) || !check_ajax_referer('ers_reservation_nonce', '_ajax_nonce', false)) {
        error_log('❌ 不正なリクエスト（nonce検証に失敗）');
        wp_send_json_error(
            esc_html__('不正なリクエストです（セキュリティチェックNG）', 'smart-booking-pro2'),
            403
        );
        return;
    }

    // データをサニタイズ  
    // Sanitize data
    $reserve_daytime = sanitize_text_field($_POST['reserve_daytime'] ?? null);
    $reservation_name = sanitize_text_field($_POST['reservation_name'] ?? null);
    $reservation_email = sanitize_email($_POST['reservation_email'] ?? null);
    $reservation_phone = sanitize_text_field($_POST['reservation_phone'] ?? null);
    $reservation_zip = sanitize_text_field($_POST['reservation_zip'] ?? null);
    $reservation_address = sanitize_text_field($_POST['reservation_address'] ?? null);
    $reservation_building = sanitize_text_field($_POST['reservation_building'] ?? null);
    $reservation_message = sanitize_textarea_field($_POST['reservation_message'] ?? null);
    $reservation_management = intval($_POST['reservation_management'] ?? 0);
    $reservation_page_title = sanitize_text_field($_POST['reservation_page_title'] ?? null);
    $reservation_auto_accept = intval($_POST['reservation_auto_accept'] ?? 0);
    $reservation_confirmation_email = intval($_POST['reservation_confirmation_email'] ?? 0);
    $event_interval_time = intval($_POST['event_interval_time'] ?? 30);
    $calendar_add_enabled = intval($_POST['calendar_add_enabled'] ?? 0);
    $post_id = intval($_POST['reservation_post_id'] ?? 0);
    $ref_eventreservation = sanitize_text_field($_POST['reserved_ref_eventreservation'] ?? '');


    // リマインダー関連  
    // Reminder-related
    $reservation_reminder_enabled = intval($_POST['reservation_reminder_enabled'] ?? 0);
    $reservation_reminder_time = intval($_POST['reservation_reminder_time'] ?? 60);
    $reservation_reminder_template = sanitize_text_field($_POST['reservation_reminder_template'] ?? 'remainder-reservation-default.txt');

    // 予約確認メールのテンプレート  
    // Reservation confirmation email template
    $confirmation_template = sanitize_text_field($_POST['reservation_confirmation_template'] ?? '');

    // 予約日と時間を適切なフォーマットに変換  
    // Convert reservation date and time to appropriate format
    $reserved_day = null;
    $reserved_time = null;

    if (!empty($reserve_daytime) && is_string($reserve_daytime) && strpos($reserve_daytime, " ") !== false) {
        list($day, $time) = explode(" ", $reserve_daytime);
    
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $day)) {
            $reserved_day = $day;
        }
    
        if (preg_match('/^\d{2}:\d{2}$/', $time)) {
            $reserved_time = $time;
        }
    }

    // 必須データ無い場合は処理を中止  
    // Abort process if required data is missing
    if (empty($reservation_name) && empty($reservation_email) && empty($reserved_day) && empty($reserved_time)) {
        error_log("ajax:全ての主要フィールドが空なのでスキップしました");
        wp_send_json_success([
            'message' => esc_html__('投稿はスキップされました（入力なし）', 'smart-booking-pro2'),
            'skipped' => true
        ]);
        return;
    }

    

    // 追加項目の取得（サニタイズ）  
    // Retrieve additional items (sanitize)
    $reserved_age = intval($_POST['reserved_age'] ?? 0);
    $reserved_gender = sanitize_text_field($_POST['reserved_gender'] ?? '未回答');
    $reserved_marital_status = sanitize_text_field($_POST['reserved_marital_status'] ?? '未回答');
    $reserved_income = intval($_POST['reserved_income'] ?? 0);
    $reserved_residence = sanitize_text_field($_POST['reserved_residence'] ?? '未選択');
    $reserved_rent = intval($_POST['reserved_rent'] ?? 0);
    $reserved_people = intval($_POST['reserved_people'] ?? 1);
    $reserved_course = sanitize_text_field($_POST['reserved_course'] ?? '未選択');
    $reserved_car_count = intval($_POST['reserved_car_count'] ?? 0);
    $reserved_hobby = sanitize_text_field($_POST['reserved_hobby'] ?? '');


    // 予約情報を `reserved-daytime` 投稿として保存  
    // Save reservation information as a `reserved-daytime` post
    $new_post = [
        'post_title'   => $reserve_daytime . " - " . $reservation_name,
        'post_status'  => 'draft', // ✅ ここを draft に変更
        'post_type'    => 'reserved-daytime',
        'post_author'  => 1,
    ];

    $post_id = wp_insert_post($new_post);

    if ($post_id) {

        $acf_fields = [
            'reserved-user-name' => $reservation_name,
            'reserved-user-email' => $reservation_email,
            'reserved-day' => $reserved_day,
            'reserved-time' => $reserved_time,
            'reserved-phone' => $reservation_phone,
            'reserved-address' => $reservation_address,
            'reserved-building' => $reservation_building,
            'reserved-message' => $reservation_message,
            'event-id' => $reservation_management,
            'reserved-zip' => $reservation_zip,
            'event-name' => $reservation_page_title,
            'reserved-auto-accept' => $reservation_auto_accept,
            'confirmation-email-enabled' => $reservation_confirmation_email,
            'reminder_enabled' => $reservation_reminder_enabled,
            'reminder_time' => $reservation_reminder_time,
            'reminder_template' => $reservation_reminder_template,
            'confirmation_template' => $confirmation_template,

            'reserved-age' => $reserved_age,
            'reserved-gender' => $reserved_gender,
            'reserved-marital-status' => $reserved_marital_status,
            'reserved-income' => $reserved_income,
            'reserved-residence' => $reserved_residence,
            'reserved-rent' => $reserved_rent,
            'reserved-people' => $reserved_people,
            'reserved-course' => $reserved_course,
            'reserved-car-count' => $reserved_car_count,
            'reserved-hobby' => $reserved_hobby,
            'reserved_ref_eventreservation' => $ref_eventreservation
        ];

        foreach ($acf_fields as $key => $value) {
            if ($value === null || $value === '') {
                error_log("スキップするACFフィールド: {$key} (値が空)");
                continue;
            }
            update_field($key, $value, $post_id);
        }
        

        // プレースホルダー用の配列を作成  
        // Create an array for placeholders
        $placeholders = [];
        foreach ($acf_fields as $key => $value) {
            $placeholders['{' . $key . '}'] = $value;
        }

        
        // 確認メール送信  
        // Send confirmation email
        if ($reservation_confirmation_email) {
            $template_dir = plugin_dir_path(__FILE__) . "../templates/confirms/";
            $template_file = $template_dir . $confirmation_template;
        
            $subject = esc_html__('【予約確認】ご予約を受け付けました', 'smart-booking-pro2'); // デフォルト件名
            $content_type = 'text/plain'; // デフォルトはプレーンテキスト（charsetはあとで追加）
        
            if (!empty($confirmation_template) && file_exists($template_file)) {
                $message = file_get_contents($template_file);
        
                // エンコーディング変換  
                // Encoding conversion
                $message = mb_convert_encoding($message, 'UTF-8', mb_detect_encoding($message, 'UTF-8, SJIS, EUC-JP, JIS, ASCII', true));
        
                // 件名の抽出＆削除  
                // Extract and remove subject
                if (preg_match('/^Subject:\s*(.+)/m', $message, $matches)) {
                    $subject = trim($matches[1]);
                    $message = preg_replace('/^Subject:\s*(.+)\n/m', '', $message);
                }
        
                // 拡張子に応じてContent-Typeを切り替え  
                // Switch Content-Type according to file extension
                $extension = pathinfo($template_file, PATHINFO_EXTENSION);
                if ($extension === 'html') {
                    $content_type = 'text/html';
                }
            } else {
                // デフォルトのメール本文  
                // Default email body
                $message = esc_html__("【予約確認】ご予約を受け付けました", 'smart-booking-pro2') . "\n\n";
                $message .= "{reserved-user-name} " . esc_html__('様', 'smart-booking-pro2') . "\n\n";
                $message .= esc_html__("この度はご予約ありがとうございます。", 'smart-booking-pro2') . "\n";
                $message .= esc_html__("予約日時:", 'smart-booking-pro2') . " {reserved-day} {reserved-time}\n";
                $message .= esc_html__("お問い合わせ内容:", 'smart-booking-pro2') . " {reserved-message}\n\n";
                $message .= esc_html__("ご不明点がございましたら、お気軽にお問い合わせください。", 'smart-booking-pro2') . "\n";
                $message .= esc_html__("▼ カレンダーに追加する:", 'smart-booking-pro2') . "\n{@calendar_links}\n";
                $message .= "------------------------------------------------------------";
            }
        
            // カレンダーリンクをメール本文に追加  
            // Add calendar link to email body
            $calendar_placeholder_key = '{@calendar_links}';
        error_log('ajax:到達7');
            if ($calendar_add_enabled && !empty($reserved_day) && !empty($reserved_time)) {
                $start_datetime = "$reserved_day $reserved_time";
                $end_datetime = date('Y-m-d H:i', strtotime($start_datetime . " +{$event_interval_time} minutes"));
        
                $calendar_url = generate_google_calendar_url($reservation_page_title, $start_datetime, $end_datetime);
        
                if ($content_type === 'text/plain') {
                    $placeholders[$calendar_placeholder_key] = "\n" . esc_html__('▼ ご自身のカレンダーに登録する：', 'smart-booking-pro2') . "\n" . $calendar_url . "\n";
                } else {
                    $placeholders[$calendar_placeholder_key] = '
                        <p style="margin-top:20px;">
                            <a href="' . esc_url($calendar_url) . '" 
                            style="display:inline-block;padding:10px 20px;background:#4285f4;color:#fff;text-decoration:none;border-radius:5px;"
                            target="_blank">
                            ' . esc_html__('Googleカレンダーに追加する', 'smart-booking-pro2') . '
                            </a>
                        </p>';
                }
            } else {
                // カレンダー追加が無効の場合は、空文字を挿入してプレースホルダーを削除  
                // If calendar addition is disabled, insert empty string and remove placeholder
                $placeholders[$calendar_placeholder_key] = '';
            }
        
            // プレースホルダー置換  
            // Placeholder replacement
            $message = str_replace(array_keys($placeholders), array_values($placeholders), $message ?? '');
            $subject = str_replace(array_keys($placeholders), array_values($placeholders), $subject ?? '');


        
            // Content-Type に charset を付けてヘッダー作成  
            // Create header with charset in Content-Type
            $headers = ['Content-Type: ' . $content_type . '; charset=UTF-8'];
        
            // メール送信  
            // Send email
            if (wp_mail($reservation_email, $subject, $message, $headers)) {
                wp_send_json_success(
                    esc_html__('予約データが保存されました', 'smart-booking-pro2')
                );
            } else {
                error_log(esc_html__("メールの送信に失敗しました", 'smart-booking-pro2'));
                wp_send_json_error(esc_html__("メールの送信に失敗しました", 'smart-booking-pro2'));
            }
        } else {
            error_log(esc_html__("予約データの保存に失敗", 'smart-booking-pro2'));
            wp_send_json_error(esc_html__("予約データの保存に失敗しました", 'smart-booking-pro2'));
        }
        

        wp_send_json_success(__('予約データが保存されました', 'smart-booking-pro2'));
    } else {
        error_log("予約データの保存に失敗");
        wp_send_json_error(
            esc_html__('予約データの保存に失敗しました', 'smart-booking-pro2')
        );
    }
    exit;
}

add_action('wp_ajax_nopriv_save_reservation_data', 'save_reservation_data');
add_action('wp_ajax_save_reservation_data', 'save_reservation_data');


function save_acf_data() {
    // 🔐 nonce 検証（エラー時は即終了）  
    // 🔐 Nonce verification (exit immediately on error)
    if (!isset($_POST['_ajax_nonce']) || !check_ajax_referer('ers_reservation_nonce', '_ajax_nonce', false)) {
        error_log('❌ 不正なリクエスト（nonce検証に失敗）');
        wp_send_json_error(
            esc_html__('不正なリクエストです（セキュリティチェックNG）', 'smart-booking-pro2'),
            403
        );
        return;
    }

    // 🔹 必要なデータが送信されているかチェック  
    // 🔹 Check if necessary data has been sent
    if (!isset($_POST['database_db']) || !isset($_POST['acf_fields'])) {
        error_log("ACFデータ保存エラー: 必須データ (database_db, acf_fields) が不足");

        wp_send_json_success([
            'skipped' => true,
            'message' => esc_html__('ACFデータが不足しているため、保存をスキップしました', 'smart-booking-pro2')
        ]);
        return;
    }


    // 🔹 送信データを取得  
    // 🔹 Retrieve submitted data
    $database_db = sanitize_text_field($_POST['database_db']); // カスタム投稿タイプ名
    $acf_fields = $_POST['acf_fields'];



if (!empty($_POST['acf_fields']) && is_array($_POST['acf_fields'])) {
    foreach ($_POST['acf_fields'] as $key => $val) {
        $type = gettype($val);
        $val_str = is_scalar($val) ? $val : json_encode($val);
        error_log("🧩 ACFフィールド → {$key} ({$type}): {$val_str}");
    }
} else {
    error_log("⚠️ acf_fields が空か無効です");
}

$field_groups = acf_get_field_groups(['post_type' => $database_db]);

foreach ($field_groups as $group) {
    error_log("📘 フィールドグループ: " . $group['title']);
    $fields = acf_get_fields($group['key']);
    foreach ($fields as $field) {
        error_log("🔧 フィールド名: {$field['name']} / タイプ: {$field['type']}");
    }
}


    $acf_fields['create_date'] = current_time('Y-m-d H:i'); // WP時刻（ローカル時間）
    $admin_user = get_user_by('ID', 1);
    $acf_fields['create_user'] = $admin_user ? $admin_user->user_login : 'admin';

    // 🔹 確認メール関連の受信データ  
    // 🔹 Received data related to confirmation email
    $reservation_confirmation_email = intval($_POST['reservation_confirmation_email'] ?? 0);
    $confirmation_template = sanitize_text_field($_POST['reservation_confirmation_template'] ?? '');
    $email_candidates = $_POST['email_candidates'] ?? []; // ← ✨ 追加（複数メール対応）


    // 🔹 新規投稿を作成  
    // 🔹 Create new post
    $new_post = array(
        'post_title'  => '新規データ (' . current_time('Y-m-d H:i:s') . ')',
        'post_status' => 'draft',
        'post_type'   => $database_db, // 指定されたカスタム投稿タイプ
        'post_author' => 1, // 管理者ユーザー (ID 1)
    );

    $post_id = wp_insert_post($new_post);

    // 🔹 投稿作成に失敗した場合の処理  
    // 🔹 Process when post creation fails
    if (is_wp_error($post_id) || !$post_id) {
        error_log("ACFデータ保存エラー: 投稿の作成に失敗しました");
        wp_send_json_error(
            esc_html__('投稿の作成に失敗しました', 'smart-booking-pro2')
        );
        return;
    }

    $email_address = ''; // ← 初期化：最初に見つかった email フィールドの値を入れる

    foreach ($acf_fields as $key => $value) {
    // 値の整形（基本はスカラ値前提）  
    // Format values (basically assumes scalar values)
    $value_safe = is_scalar($value) ? sanitize_text_field($value) : '';
    
    // 🔹 先に保存（これがないと get_field_object が false になることがある）  
    // 🔹 Save first (without this, get_field_object may return false)
    update_field($key, $value_safe, $post_id);
    error_log("ACFデータ保存: {$key} = {$value_safe}");

    // 🔍 保存後に取得（これならACFの構造が有効になる）  
    // 🔍 Retrieve after saving (this enables the ACF structure)
    $field_object = get_field_object($key, $post_id);
    if (!is_array($field_object)) {
        error_log("▼ get_field_object 取得失敗（保存後でもNG）：{$key}");
        continue;
    }

    $field_type = $field_object['type'];
    error_log("◎ フィールド情報：{$key} のタイプは {$field_type}");

    // ✉️ email型なら検証  
    // ✉️ Validate if type is email
    if ($field_type === 'email') {
        error_log("→ email型フィールド：{$key}、値 = {$value_safe}");
        if (is_email($value_safe)) {
            $email_address = $value_safe;
            error_log("★ メールアドレス抽出成功：{$email_address}（フィールド名：{$key}）");
        } else {
            error_log("× is_email() で false：{$value_safe}");
        }
    }
}


    error_log("★ reservation_confirmation_email = {$reservation_confirmation_email}");
    error_log("★ confirmation_template = {$confirmation_template}");
    if (!empty($email_candidates)) {
        foreach ($email_candidates as $candidate) {
            $name = $candidate['name'] ?? '';
            $value = $candidate['value'] ?? '';
            error_log("★ メール候補：name = {$name}, value = {$value}");
        }
    } else {
        error_log("★ メール候補なし（email_candidatesが空）");
    }

    if ($reservation_confirmation_email === 1) {

        if (empty($email_candidates)) {
            error_log("📭 確認メールスキップ：候補メールアドレスが空のため送信なし");
            error_log("ACFデータ保存成功: 投稿ID = {$post_id}");
            wp_send_json_success(
                esc_html__('ACFデータが保存されました', 'smart-booking-pro2')
            );
            return;
        }

        $template_path = plugin_dir_path(__FILE__) . '../templates/confirms/' . $confirmation_template;
    
        if (file_exists($template_path)) {
            $template_content = file($template_path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
            if ($template_content && strpos($template_content[0], 'Subject:') === 0) {

                $subject_line = array_shift($template_content) ?? '';
                $subject = trim(str_replace('Subject:', '', $subject_line));

                $body_template = implode("\n", $template_content);
    
                // HTMLか判定  
                // Check if it is HTML
                $is_html = stripos($subject_line, 'TMHTML') !== false;
                $headers = ['Content-Type: ' . ($is_html ? 'text/html' : 'text/plain') . '; charset=UTF-8'];
    
                // 🔁 候補アドレスすべてにループ送信  
                // 🔁 Loop send to all candidate addresses
                foreach ($email_candidates as $candidate) {
                    $email = $candidate['value'] ?? '';
                    if (!is_email($email)) {
                        error_log("× 無効なメールアドレス形式（スキップ）：{$email}");
                        continue;
                    }
    
                    // 件名・本文それぞれのコピーを作成（毎回リセット）  
                    // Create copies of subject and body respectively (reset each time)
                    $subject_replaced = $subject;
                    $body_replaced = $body_template;
    
                    foreach ($acf_fields as $key => $value) {
                        $placeholder = '{' . $key . '}';
                        $safe_value = sanitize_text_field($value);
                        $body_replaced = str_replace($placeholder, $safe_value, $body_replaced ?? '');
                        $subject_replaced = str_replace($placeholder, $safe_value, $subject_replaced ?? '');

                    }
    
                    // 特殊タグ（例：{@calendar_links}）除去や置換  
                    // Remove or replace special tags (e.g., {@calendar_links})
                    $body_replaced = str_replace('{@calendar_links}', '', $body_replaced ?? '');
    
                    // Send mail
                    $sent = wp_mail($email, $subject_replaced, $body_replaced, $headers);
    
                    if ($sent) {
                        error_log("✅ 確認メール送信成功：{$email}");
                    } else {
                        error_log("❌ 確認メール送信失敗：{$email}");
                    }
                }
            } else {
                error_log("⚠️ テンプレートの1行目に 'Subject:' が見つかりません: {$template_path}");
            }
        } else {
            error_log("⚠️ 確認メールテンプレートが存在しません: {$template_path}");
        }
    }
    

    error_log("ACFデータ保存成功: 投稿ID = {$post_id}");
    wp_send_json_success(__('ACFデータが保存されました', 'smart-booking-pro2'));


}


add_action('wp_ajax_nopriv_save_acf_data', 'save_acf_data');
add_action('wp_ajax_save_acf_data', 'save_acf_data');


function fetch_reservation_data() {
    // 🔐 必須チェック
    if (empty($_POST['event_id']) || empty($_POST['selected_date'])) {
        wp_send_json_error(
            esc_html__('必須データが不足しています', 'smart-booking-pro2')
        );
        return;
    }

    // 🔄 値の取得・整形
    $event_id = sanitize_text_field($_POST['event_id']);
    $selected_date_raw = sanitize_text_field($_POST['selected_date']);
    $selected_date = str_replace('-', '', $selected_date_raw); // 2025-06-28 → 20250628

    // 🎯 条件に合致する投稿だけ取得
    $query = new WP_Query([
        'post_type'      => 'reserved-daytime',
        'post_status'    => 'any',
        'posts_per_page' => -1,
        'meta_query'     => [
            'relation' => 'AND',
            [
                'key'     => 'event-id',
                'value'   => $event_id,
                'compare' => '=',
                'type'    => 'CHAR'
            ],
            [
                'relation' => 'OR',
                [
                    'key'     => 'reserved-day',
                    'value'   => $selected_date, // 20250628
                    'compare' => '=',
                    'type'    => 'CHAR'
                ],
                [
                    'key'     => 'reserved-day',
                    'value'   => $selected_date_raw, // 2025-06-28
                    'compare' => '=',
                    'type'    => 'CHAR'
                ]
            ]
        ]
    ]);

    $reservations = [];

    while ($query->have_posts()) {
        $query->the_post();
        $reservations[] = [
            'reserved_time' => get_field('reserved-time'),
            'auto_accept'   => get_field('reserved-auto-accept') ? "1" : "0"
        ];
    }

    wp_reset_postdata();
    wp_send_json_success($reservations);
}

add_action('wp_ajax_nopriv_fetch_reservation_data', 'fetch_reservation_data');
add_action('wp_ajax_fetch_reservation_data', 'fetch_reservation_data');


// 指定カスタム投稿タイプのACFフィールドを取得する  
// Retrieve ACF fields for the specified custom post type
function get_acf_fields_ajax() {

    error_log('gacf：到達0');
    if (!isset($_POST['post_type']) || empty($_POST['post_type'])) {
        wp_send_json_error([
            'message' => esc_html__('投稿タイプが指定されていません。', 'smart-booking-pro2')
        ]);
        wp_die();
    }

    $post_type = sanitize_text_field($_POST['post_type']);
    $field_groups = acf_get_field_groups(['post_type' => $post_type]);

    $fields = [];

    if (!is_array($field_groups)) {
        wp_send_json_error([
            'message' => esc_html__('フィールドグループが取得できませんでした。', 'smart-booking-pro2')
        ]);
        wp_die();
    }

    foreach ($field_groups as $group) {
        if (!isset($group['key'])) {
            error_log("⚠️ フィールドグループに 'key' が存在しません（post_type: {$post_type}）");
            continue;
        }

        $group_fields = acf_get_fields($group['key']);
        if (!is_array($group_fields)) continue;

        foreach ($group_fields as $field) {
            if (isset($field['name'])) {
                $fields[] = $field['name'];
            } else {
                error_log("⚠️ ACFフィールドに 'name' が存在しません（group key: {$group['key']}）");
            }
        }
    }

    wp_send_json_success($fields);
    wp_die();
}


// AJAX エンドポイント登録  
// Register AJAX endpoint
add_action('wp_ajax_get_acf_fields', 'get_acf_fields_ajax');
add_action('wp_ajax_nopriv_get_acf_fields', 'get_acf_fields_ajax'); // 未ログインユーザーでもOK

// 秘密キー更新  
// Update secret key
add_action('wp_ajax_sbp2_save_jwt_secret', function () {

    $jwt_secret = sanitize_text_field($_POST['jwt_secret'] ?? '');
    if (empty($jwt_secret)) {
        wp_send_json_error(esc_html__('JWT Secretが空です。', 'smart-booking-pro2'));
    }

    update_option('sbp2_jwt_secret', $jwt_secret);
    wp_send_json_success();
});

// サブスクのカスタマーIDと登録メールアドレスの取得  
// Retrieve subscription customer ID and registered email address
add_action('wp_ajax_sbp2_get_stripe_info', function () {

    $customer_id = get_option('sbp2_stripe_customer_id');
    $email       = get_option('sbp2_stripe_email');

    if (empty($customer_id)) {
        wp_send_json_error(esc_html__('StripeのCustomer IDが未設定です', 'smart-booking-pro2'));
    }

    wp_send_json_success([
        'customer_id' => $customer_id,
        'email'       => $email,
    ]);
});

