<?php
if (!defined('ABSPATH')) {
    exit;
}

function smart_booking_export_csv() {

    // ✅ PROユーザー制限（最重要）  
    // ✅ PRO user restriction (highest priority)
    if (!function_exists('sbp2_is_pro_user') || !(sbp2_is_pro_user() || sbp2_is_enterprise_user())) {
        wp_die(esc_html__('この機能は PRO プラン以上でご利用いただけます。', 'smart-booking-pro2'));
    }

    if (!isset($_GET['post_type'])) {
        wp_die(esc_html__('エラー: 投稿タイプが指定されていません。', 'smart-booking-pro2'));
    }

    $post_type = sanitize_text_field($_GET['post_type']);
    $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : '';
    $end_date = isset($_GET['end_date']) ? sanitize_text_field($_GET['end_date']) : '';

    // ✅ **ACF フィールドの取得**  
    // ✅ **Retrieve ACF fields**
    $saved_acf_data = get_option('smart_booking_acf_data', array());
    $saved_acf_db = get_option('smart_booking_acf_db', array());

    $acf_fields = [];

    // ✅ `reserved-daytime` の ACF フィールドを手動で定義  
    // ✅ Manually define ACF fields for `reserved-daytime`
    if ($post_type === 'reserved-daytime') {
        $acf_fields = [
            'event-id', 'event-name', 'reserved-user-name', 'reserved-user-email',
            'reserved-day', 'reserved-time', 'reserved-phone', 'reserved-zip',
            'reserved-address', 'reserved-building', 'reserved-message',
            'reserved-auto-accept', 'confirmation-email-enabled', 'confirmation_template',
            'reminder_enabled', 'reminder_time', 'reminder_template',
            'reserved-age', 'reserved-gender', 'reserved-marital-status',
            'reserved-income', 'reserved-residence', 'reserved-rent', 'reserved-people',
            'reserved-course', 'reserved-car-count', 'reserved-hobby'
        ];
    } else {
        // ✅ 他の投稿タイプの場合は通常通り ACF フィールドを取得  
        // ✅ Retrieve ACF fields normally for other post types
        foreach ($saved_acf_data as $data) {
            if ($data['post_type'] === $post_type) {
                foreach ($data['acf_fields'] as $field) {
                    $acf_fields[] = $field[0]; // フィールド名のみ取得
                }
                break;
            }
        }

        foreach ($saved_acf_db as $data) {
            if ($data['post_type'] === $post_type) {
                foreach ($data['acf_fields'] as $field) {
                    $acf_fields[] = $field[0]; // フィールド名のみ取得
                }
                break;
            }
        }
    }

    // ✅ **重複するフィールドを除外**  
    // ✅ **Exclude duplicate fields**
    $acf_fields = array_unique($acf_fields);

    if (empty($acf_fields)) {
        wp_die(esc_html__('エラー: 指定された投稿タイプの ACF フィールドが見つかりません。', 'smart-booking-pro2'));
    }

    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="export_' . $post_type . '.csv"');

    $output = fopen('php://output', 'w');
    fputs($output, "\xEF\xBB\xBF"); // ✅ Excel用にUTF-8 BOMを追加

    // ✅ **ヘッダー行を作成（基本情報 + ACFフィールド）**  
    // ✅ **Create header row (basic info + ACF fields)**
    $header = array_merge(['ID', 'タイトル', '登録日'], $acf_fields);
    fputcsv($output, $header);

    // ✅ **フィルタリング用の meta_query を作成**  
    // ✅ **Create meta_query for filtering**
    $meta_query = [];

    if ($post_type === 'reserved-daytime' && (!empty($start_date) || !empty($end_date))) {
        $date_filter = ['relation' => 'AND'];

        if (!empty($start_date)) {
            $date_filter[] = [
                'key'     => 'reserved-day',
                'value'   => $start_date,
                'compare' => '>=',
                'type'    => 'DATE'
            ];
        }

        if (!empty($end_date)) {
            $date_filter[] = [
                'key'     => 'reserved-day',
                'value'   => $end_date,
                'compare' => '<=',
                'type'    => 'DATE'
            ];
        }

        $meta_query[] = $date_filter;
    }

    // ✅ **対象の投稿を取得**  
    // ✅ **Retrieve target posts**
    $query_args = array(
        'post_type'      => $post_type,
        'posts_per_page' => -1,
        'post_status'    => 'any',
        'meta_query'     => !empty($meta_query) ? $meta_query : null
    );

    $query = new WP_Query($query_args);

    if (!$query->have_posts()) {
        fputcsv($output, [
            esc_html__('エラー', 'smart-booking-pro2'),
            esc_html__('データがありません。', 'smart-booking-pro2')
        ]);
        fclose($output);
        exit;
    }

    // ✅ **投稿データをループして出力**  
    // ✅ **Loop through post data and output**
    while ($query->have_posts()) {
        $query->the_post();
        $row = [
            get_the_ID(),
            get_the_title(),
            get_the_date('Y-m-d'),
        ];

        foreach ($acf_fields as $field) {
            $row[] = get_post_meta(get_the_ID(), $field, true) ?: ''; // 値がなければ空白
        }

        fputcsv($output, $row);
    }

    fclose($output);
    exit;
}

add_action('wp_ajax_smart_booking_export_csv', 'smart_booking_export_csv');


// 予約リストCSVエクスポート処理（全ACFフィールド対応）  
// Reservation list CSV export process (supports all ACF fields)
function ers_export_reservation_csv() {

    $args = ['post_type' => 'reserved-daytime', 'posts_per_page' => -1,'post_status' => 'any', ];

    if (!empty($_GET['filter_date'])) {
        $args['meta_query'][] = [
            'key' => 'reserved-day',
            'value' => $_GET['filter_date'],
            'compare' => '='
        ];
    }
    if (!empty($_GET['filter_event_id'])) {
        $args['meta_query'][] = [
            'key' => 'event-id',
            'value' => $_GET['filter_event_id'],
            'compare' => '='
        ];
    }

    $reservations = get_posts($args);
    if (empty($reservations)) {
        wp_die('予約データがありません。');
    }

    // ヘッダーを指定  
    // Specify headers
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="reservation_list.csv"');
    header("Pragma: no-cache");
    header("Expires: 0");

    // 出力バッファを開始  
    // Start output buffering
    $output = fopen('php://output', 'w');

    // UTF-8 BOMを追加（Excelで文字化けしないように）  
    // Add UTF-8 BOM (to prevent garbled text in Excel)
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

    // **ACFフィールドのリストを取得（動的対応）**  
    // **Retrieve list of ACF fields (dynamic support)**
    $acf_fields = acf_get_fields('group_reserved_daytime');
    if (!$acf_fields) {
        wp_die('ACFフィールドが見つかりません。');
    }

    // **ヘッダー行を作成**  
    // **Create header row**
    $csv_headers = ['予約ID']; // 最初のカラム（投稿ID）
    foreach ($acf_fields as $field) {
        $csv_headers[] = $field['label']; // ラベル名をCSVのヘッダーとして使用
    }
    fputcsv($output, $csv_headers);

    // **データ行を作成**  
    // **Create data rows**
    foreach ($reservations as $reservation) {
        $fields = get_fields($reservation->ID);
        $row = [$reservation->ID]; // 予約IDを最初の列に追加

        foreach ($acf_fields as $field) {
            $field_name = $field['name'];
            $row[] = $fields[$field_name] ?? ''; // 値がない場合は空欄
        }
        
        fputcsv($output, $row);
    }

    fclose($output);
    exit();
}

// CSVエクスポート処理をフック登録  
// Register hook for CSV export processing
add_action('admin_post_export_reservation_csv', 'ers_export_reservation_csv');
add_action('admin_post_nopriv_export_reservation_csv', 'ers_export_reservation_csv');


//CSVアップロード時
//フィールド名を取得する関数
function get_acf_field_names_by_post_type($post_type) {
    $names = [];

    $field_groups = acf_get_field_groups(['post_type' => $post_type]);
    if (!$field_groups) return $names;

    foreach ($field_groups as $group) {
        $fields = acf_get_fields($group['key']);
        if (is_array($fields)) {
            foreach ($fields as $field) {
                if (!empty($field['name'])) {
                    $names[] = $field['name'];
                }
            }
        }
    }

    return $names;
}

// CSVアップロード処理  
// CSV upload processing
add_action('admin_init', 'smart_booking_handle_csv_upload');

function smart_booking_handle_csv_upload() {

    if (!isset($_POST['submit_csv_upload'])) return;

    if (!isset($_FILES['csv_file']) || empty($_FILES['csv_file']['tmp_name'])) {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>CSVファイルが選択されていません。</p></div>';
        });
        return;
    }

    $file = $_FILES['csv_file']['tmp_name'];
    $post_type = sanitize_text_field($_POST['target_post_type']);

    // 投稿タイトルに使う列のインデックス取得（複数対応）
    $title_column_indexes = [];
    if (isset($_POST['title_column'])) {
        if (is_array($_POST['title_column'])) {
            $title_column_indexes = array_map('intval', $_POST['title_column']);
        } elseif (is_string($_POST['title_column'])) {
            $title_column_indexes = [intval($_POST['title_column'])];
        }
    }

    if (empty($title_column_indexes)) {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>投稿タイトルに使う列を選択してください。</p></div>';
        });
        return;
    }

    if (empty($post_type)) {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>保存先の投稿タイプが指定されていません。</p></div>';
        });
        return;
    }

    $mime = mime_content_type($file);
    if ($mime !== 'text/plain' && $mime !== 'text/csv') {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p>CSVファイル形式のみ対応しています。</p></div>';
        });
        return;
    }

    $handle = fopen($file, 'r');
    if (!$handle) return;

    $header = fgetcsv($handle);
    if (!$header) return;

    // 🔽 BOM除去追加
    if (!empty($header[0])) {
        $header[0] = preg_replace('/^\xEF\xBB\xBF/', '', $header[0]);
    }

    // ACFフィールドの存在チェック
    $acf_field_names = get_acf_field_names_by_post_type($post_type);
    $missing = array_diff($header, $acf_field_names);


    if (!empty($missing)) {
        add_action('admin_notices', function () use ($missing) {
            echo '<div class="notice notice-error"><p>以下のフィールドがACFに存在しません: ' . implode(', ', $missing) . '</p></div>';
        });
        return;
    }

    // 投稿登録処理
    $imported = 0;
    while (($row = fgetcsv($handle)) !== false) {
        // 投稿タイトルの生成
        $title_parts = [];
        foreach ($title_column_indexes as $index) {
            if (isset($row[$index])) {
                $title_parts[] = trim($row[$index]);
            }
        }

        $title = implode(' ', $title_parts);
        if (empty($title)) {
            $title = 'CSV Imported - ' . current_time('mysql');
        }

        // 投稿作成
        $post_id = wp_insert_post([
            'post_type'   => $post_type,
            'post_status' => 'draft',
            'post_title'  => sanitize_text_field($title),
        ]);

        if (is_wp_error($post_id)) continue;

        // ACFフィールドに値を登録
        foreach ($header as $index => $field_name) {
            $field_value = $row[$index] ?? '';

            // 必要に応じてサニタイズ強化
            $sanitized = sanitize_textarea_field($field_value); // 必要なら switch($field_type) で分岐

            update_field($field_name, $sanitized, $post_id);
        }

        $imported++;
    }

    fclose($handle);

    $redirect_url = add_query_arg('csv_imported', $imported, wp_get_referer());
    wp_redirect($redirect_url);
    exit;

}
add_action('admin_notices', function () {
    if (isset($_GET['csv_imported'])) {
        $count = intval($_GET['csv_imported']);
        echo '<div class="notice notice-success"><p>' . esc_html($count) . ' 件のデータをインポートしました。</p></div>';
    }
});



function get_acf_field_keys_by_post_type($post_type) {
    if (!function_exists('acf_get_field_groups') || !function_exists('acf_get_fields')) return [];

    $field_keys = [];
    $groups = acf_get_field_groups(['post_type' => $post_type]);

    foreach ($groups as $group) {
        $fields = acf_get_fields($group['key']);
        if ($fields) {
            foreach ($fields as $field) {
                $field_keys[] = $field['name'];
            }
        }
    }

    return $field_keys;
}