<?php
if (!defined('ABSPATH')) {
    exit;
}

// ✅ 投稿タイプ & ACF フィールドを `init` フックで登録する  
// ✅ Register post types & ACF fields on the `init` hook
function smart_booking_register_custom_post_types_and_acf() {
    $saved_data_acf = get_option('smart_booking_acf_data', []);
    $saved_data_db  = get_option('smart_booking_acf_db', []);

    if (!is_array($saved_data_acf)) {
        $saved_data_acf = [];
    }
    if (!is_array($saved_data_db)) {
        $saved_data_db = [];
    }

    $merged_data = array_merge($saved_data_acf, $saved_data_db);

    if (!empty($merged_data)) {
        error_log('🔍 登録対象のデータ一覧（merged_data）:');
        error_log(print_r($merged_data, true));

        foreach ($merged_data as $index => $data) {
            $post_type_name = $data['post_type'] ?? '';
            $acf_group_key  = $data['acf_group'] ?? '';
            $acf_fields     = $data['acf_fields'] ?? [];

            error_log("✅ {$index}: 投稿タイプ登録中 → {$post_type_name}");
            error_log("┗ ACFグループキー → {$acf_group_key}");

            register_post_type($post_type_name, [
                'labels' => [
                    'name'          => $post_type_name,
                    'singular_name' => $post_type_name
                ],
                'public'             => false,
                'publicly_queryable' => false,
                'exclude_from_search'=> true,
                'has_archive'        => false,
                'show_ui'            => true,
                'show_in_menu'       => false,
                'capability_type'    => 'post',
                'supports'           => ['title'],
                'menu_position'      => 5,
                'rewrite'            => false,
            ]);

            if (function_exists('acf_add_local_field_group') && !empty($acf_fields)) {
                $fields = [];

                foreach ($acf_fields as $f_index => $field) {
                    $name  = $field['name']  ?? ($field[0] ?? '');
                    $type  = $field['type']  ?? ($field[2] ?? '');
                    $label = $field['label'] ?? ($field[1] ?? ucfirst($name));

                    if ($name && $type) {
                        $fields[] = [
                            'key'   => 'sbp2-fld-' . sanitize_key($name),
                            'label' => sanitize_text_field($label),
                            'name'  => sanitize_key($name),
                            'type'  => sanitize_key($type),
                        ];

                        error_log("  ┣ フィールド{$f_index}: name={$name}, label={$label}, type={$type}");
                    } else {
                        error_log("  ┣ ⚠️ フィールド{$f_index}が無効（name/typeが空）");
                    }
                }

                if (!empty($fields)) {
                    acf_add_local_field_group([
                        'key'      => $acf_group_key,
                        'title'    => $data['acf_group_label'] ?? $acf_group_key,
                        'fields'   => $fields,
                        'location' => [
                            [
                                [
                                    'param'    => 'post_type',
                                    'operator' => '==',
                                    'value'    => $post_type_name,
                                ],
                            ],
                        ],
                    ]);

                    error_log("✅ フィールドグループ登録完了 → {$acf_group_key}");
                } else {
                    error_log("⚠️ フィールドが空のため登録スキップ → {$acf_group_key}");
                }
            }
        }
    }
}
add_action('init', 'smart_booking_register_custom_post_types_and_acf');






// ✅ 「エディタ（本文）」を非表示にする  
// ✅ Hide the "Editor (Content)" field
function smart_booking_hide_editor() {
    global $post;
    if ($post && strpos($post->post_type, 'sbp2-typ-') === 0) {
        remove_post_type_support($post->post_type, 'editor'); // ✅ 本文エディタ削除
        remove_post_type_support($post->post_type, 'excerpt'); // ✅ 抜粋削除
        remove_post_type_support($post->post_type, 'thumbnail'); // ✅ アイキャッチ削除
        remove_post_type_support($post->post_type, 'custom-fields'); // ✅ カスタムフィールド削除
    }
}
add_action('admin_init', 'smart_booking_hide_editor');

// ✅ `flush_rewrite_rules()` を `init` で実行（投稿タイプのリライトルールを更新）  
// ✅ Run `flush_rewrite_rules()` on `init` to update post type rewrite rules
function smart_booking_flush_rewrite_rules() {
    flush_rewrite_rules();
}
add_action('init', 'smart_booking_flush_rewrite_rules');


// ✅ 投稿タイプ & ACF の作成処理  
// ✅ Process to create post types & ACF fields
function smart_booking_handle_generation() {

    if (!isset($_POST['create_db']) && !isset($_POST['generate_post_acf']) && !isset($_POST['db-create-button'])) {
        return;
    }

    if (get_transient('smart_booking_handle_generation_lock')) {
        error_log('⚠️ 二重送信防止ロック: スキップ');
        return;
    }
    set_transient('smart_booking_handle_generation_lock', true, 5);

    // ✅ POSTデータ取得＆整形
    $raw_post_type_name  = substr(sanitize_text_field($_POST['post_type_name']), 0, 10);
    $post_type_label     = sanitize_text_field($_POST['post_type_label']);
    $acf_group_name      = substr(sanitize_text_field($_POST['acf_group_name']), 0, 10);
    $acf_group_label     = sanitize_text_field($_POST['acf_group_label']);

    $acf_field_names  = $_POST['acf_field_name']  ?? [];
    $acf_field_types  = $_POST['acf_field_type']  ?? [];
    $acf_field_labels = $_POST['acf_field_label'] ?? []; // ※ まだ未使用なら空配列でもOK

    $post_type_name = 'sbp2-typ-' . $raw_post_type_name;
    $acf_group_key  = 'sbp2-grp-' . $acf_group_name;

    // ✅ ACFフィールド構築（連想配列で）
    $custom_acf_fields = [];

    foreach ($acf_field_names as $i => $name) {
        $type  = $acf_field_types[$i]  ?? '';
        $label = $acf_field_labels[$i] ?? ucfirst($name);

        if ($name && $type) {
            $custom_acf_fields[] = [
                'name'  => sanitize_key($name),
                'type'  => sanitize_key($type),
                'label' => sanitize_text_field($label),
            ];
        }
    }

    // ✅ 固定フィールド（連想配列形式で）
    $fixed_acf_fields = [
        ['name' => 'create_date', 'type' => 'date_picker', 'label' => '作成日'],
        ['name' => 'create_user', 'type' => 'text',        'label' => '作成者'],
    ];

    $all_acf_fields = array_merge($custom_acf_fields, $fixed_acf_fields);

    // ✅ 保存処理（重複チェックしてから追加）
    $saved_data = get_option('smart_booking_acf_data', []);
    if (!is_array($saved_data)) {
        $saved_data = [];
    }

    // ✅ すでに同じ post_type がある場合は除外
    $saved_data = array_filter($saved_data, function ($item) use ($post_type_name) {
        return isset($item['post_type']) && $item['post_type'] !== $post_type_name;
    });

    // ✅ 新しいデータを末尾に追加
    $saved_data[] = [
        'post_type'        => $post_type_name,
        'post_type_label'  => $post_type_label,
        'acf_group'        => $acf_group_key,
        'acf_group_label'  => $acf_group_label,
        'acf_fields'       => $all_acf_fields,
    ];
    update_option('smart_booking_acf_data', $saved_data);

    error_log("✅ 投稿タイプ保存: {$post_type_name}");

    wp_redirect(admin_url('admin.php?page=smart_booking_db&created=1'));
    exit;
}
add_action('admin_init', 'smart_booking_handle_generation');




// 🔹 **新規投稿時に自動で create_date と create_user を設定**  
// 🔹 **Automatically set create_date and create_user on new post creation**
function smart_booking_set_default_acf_values($post_id, $post, $update) {
    if ($update) {
        return; // 更新時はスキップ（新規投稿時のみ対象）
    }

    // ACFの関数が使えない環境ではスキップ
    if (!function_exists('update_field')) {
        return;
    }

    // 投稿ステータスが draft 以外ならスキップ（auto-draft などは除外）
    if (get_post_status($post_id) !== 'draft') {
        return;
    }

    $post_type = get_post_type($post_id);
    $saved_data = get_option('smart_booking_acf_data', array());

    if (!is_array($saved_data)) {
        return;
    }

    foreach ($saved_data as $data) {
        if ($data['post_type'] === $post_type) {

            // ✅ create_date を Ymd 形式で保存
            update_field('create_date', date('Ymd'), $post_id);

            // ✅ ログインユーザー名を保存
            $current_user = wp_get_current_user();
            if ($current_user && $current_user->exists()) {
                update_field('create_user', $current_user->user_login, $post_id);
            }

            break; // 一致した投稿タイプのみ処理
        }
    }
}
add_action('save_post', 'smart_booking_set_default_acf_values', 10, 3);


// csv　→　データベース化  
// Convert CSV to database entries
function smart_booking_import_csv_data($post_type, $csv_file) {

    if (!file_exists($csv_file)) {
        error_log(__('❌ CSVファイルが見つかりません:', 'smart-booking-pro2') . ' [' . $csv_file . ']');
        return;
    }

    // 📚 その投稿タイプに紐づくフィールドリストを取得
    $saved_data_acf = get_option('smart_booking_acf_data', []);
    $saved_data_db  = get_option('smart_booking_acf_db', []);
    $all_saved_data = array_merge($saved_data_acf, $saved_data_db);

    $acf_fields = [];
    foreach ($all_saved_data as $entry) {
        if (isset($entry['post_type']) && $entry['post_type'] === $post_type) {
            if (!empty($entry['acf_fields'])) {
                $acf_fields = $entry['acf_fields'];
            }
            break;
        }
    }

    if (empty($acf_fields)) {
        error_log(__('⚠️ 指定された投稿タイプにフィールド情報が見つかりません', 'smart-booking-pro2'));
        return;
    }

    if (($handle = fopen($csv_file, "r")) !== FALSE) {
        $headers = fgetcsv($handle); // 1行目ヘッダー

        while (($row = fgetcsv($handle)) !== FALSE) {
            error_log('📄 CSVデータ: ' . print_r($row, true));

            // ✅ 投稿タイトルは1列目を仮で使う（必要なら改良）
            $post_title = isset($row[0]) ? sanitize_text_field($row[0]) : '(無題)';

            $post_id = wp_insert_post([
                'post_type'   => $post_type,
                'post_title'  => $post_title,
                'post_status' => 'draft',
            ]);

            if ($post_id && !is_wp_error($post_id)) {
                // ✅ フィールドとCSVデータをマッピングして保存
                foreach ($acf_fields as $index => $field_info) {
                    $field_name = $field_info[0] ?? '';
                    if (!empty($field_name) && isset($row[$index])) {
                        update_field($field_name, sanitize_text_field($row[$index]), $post_id);
                    }
                }
                error_log(__('✅ 投稿作成成功: ID', 'smart-booking-pro2') . ' ' . $post_id);
            } else {
                error_log(__('❌ 投稿作成失敗:', 'smart-booking-pro2') . ' ' . print_r($post_id, true));
            }
        }
        fclose($handle);
    } else {
        error_log(__('❌ CSVファイルの読み込みに失敗', 'smart-booking-pro2'));
    }
}



function smart_booking_delete_type() {
    if (isset($_POST['delete_post_type'])) {
        $delete_post_type = sanitize_text_field($_POST['delete_post_type']);

        // ✅ `smart_booking_acf_data` の取得
        $saved_acf_data = get_option('smart_booking_acf_data', array());
        if (!is_array($saved_acf_data)) {
            $saved_acf_data = array();
        }

        // ✅ `smart_booking_acf_db` の取得
        $saved_acf_db = get_option('smart_booking_acf_db', array());
        if (!is_array($saved_acf_db)) {
            $saved_acf_db = array();
        }

        // **削除対象のデータを検索し、処理**
        $data_found = false;

        // ✅ `smart_booking_acf_data` から削除
        foreach ($saved_acf_data as $index => $data) {
            if ($data['post_type'] === $delete_post_type) {
                $data_found = true;
                // **投稿データの削除**
                $posts = get_posts(array(
                    'post_type' => $delete_post_type,
                    'numberposts' => -1
                ));
                foreach ($posts as $post) {
                    wp_delete_post($post->ID, true);
                }

                // **投稿タイプの登録解除**
                unregister_post_type($delete_post_type);

                // **ACFグループを削除**
                if (function_exists('acf_remove_local_field_group')) {
                    acf_remove_local_field_group($data['acf_group']);
                }

                // **配列からデータを削除**
                unset($saved_acf_data[$index]);
            }
        }

        // ✅ `smart_booking_acf_db` から削除
        foreach ($saved_acf_db as $index => $data) {
            if ($data['post_type'] === $delete_post_type) {
                $data_found = true;
                unset($saved_acf_db[$index]);
            }
        }

        // **更新**
        update_option('smart_booking_acf_data', array_values($saved_acf_data));
        update_option('smart_booking_acf_db', array_values($saved_acf_db));

        // **削除完了メッセージ**
        if ($data_found) {
            echo '<div class="updated"><p>' . 
                sprintf(
                    esc_html__('投稿タイプ「%s」と関連データを削除しました。', 'smart-booking-pro2'),
                    esc_html($delete_post_type)
                ) . '</p></div>';
        } else {
            echo '<div class="error"><p>' . 
                sprintf(
                    esc_html__('投稿タイプ「%s」が見つかりませんでした。', 'smart-booking-pro2'),
                    esc_html($delete_post_type)
                ) . '</p></div>';
        }

    }
}
add_action('admin_init', 'smart_booking_delete_type');



// ✅ DB作成処理（カスタム投稿タイプ + ACF フィールド）  
// ✅ Database creation process (custom post type + ACF fields)
function smart_booking_create_database() {
    if (isset($_POST['db-create-button'])) {

        // ✅ 投稿タイプ・グループの内部名と表示名を受け取る
        $post_type_name  = sanitize_text_field($_POST['post_type_name']);
        $post_type_label = sanitize_text_field($_POST['post_type_label']);
        $acf_group_name  = sanitize_text_field($_POST['acf_group_name']);
        $acf_group_label = sanitize_text_field($_POST['acf_group_label']);

        $acf_field_names  = $_POST['acf_field_name'];
        $acf_field_labels = $_POST['acf_field_label'];
        $acf_field_types  = $_POST['acf_field_type'];

        // ✅ 入力制限: 内部名は半角英数字10文字に制限
        $post_type_name  = substr($post_type_name, 0, 10);
        $acf_group_name  = substr($acf_group_name, 0, 10);

        // ✅ 投稿タイプ名のネーミングルール適用
        $post_type = 'sbp2-' . $post_type_name;
        $acf_group_key = 'sbp2-grp-' . $acf_group_name;

        // ✅ ユーザーが入力したACFフィールドをまとめる
        $custom_acf_fields = [];
        foreach ($acf_field_names as $index => $name) {
            $custom_acf_fields[] = [
                sanitize_text_field($name), 
                isset($acf_field_labels[$index]) ? sanitize_text_field($acf_field_labels[$index]) : '',
                sanitize_text_field($acf_field_types[$index] ?? 'text')
            ];
        }

        // ✅ 固定ACFフィールド（内部名だけ）
        $fixed_acf_fields = [
            ['event-id', __('イベントID', 'smart-booking-pro2'), 'number'],
            ['event-name', __('イベント名', 'smart-booking-pro2'), 'text'],
            ['reserved-auto-accept', __('自動承認', 'smart-booking-pro2'), 'true_false'],
            ['confirmation-email-enabled', __('確認メール送信', 'smart-booking-pro2'), 'true_false'],
            ['confirmation_template', __('確認テンプレート', 'smart-booking-pro2'), 'select'],
            ['reminder_enabled', __('リマインダー送信', 'smart-booking-pro2'), 'true_false'],
            ['reminder_time', __('リマインダー時刻', 'smart-booking-pro2'), 'number'],
            ['reminder_template', __('リマインダーテンプレート', 'smart-booking-pro2'), 'select'],
        ];


        // ✅ 全ACFフィールドを統合
        $all_acf_fields = array_merge($custom_acf_fields, $fixed_acf_fields);

        // ✅ データ保存
        $saved_data = get_option('smart_booking_acf_db', []);
        if (!is_array($saved_data)) {
            $saved_data = [];
        }

        $saved_data[] = [
            'post_type'       => $post_type,
            'post_type_label' => $post_type_label,
            'acf_group'       => $acf_group_key,
            'acf_group_label' => $acf_group_label,
            'acf_fields'      => $all_acf_fields,
        ];

        update_option('smart_booking_acf_db', $saved_data);

        // ✅ 成功メッセージ
        echo '<div class="updated"><p>' . sprintf(
            esc_html__('✅ 投稿タイプ「%s」とACFグループ「%s」を作成しました！', 'smart-booking-pro2'),
            esc_html($post_type_label),
            esc_html($acf_group_label)
        ) . '</p></div>';

    }
}


add_action('admin_init', 'smart_booking_create_database');


// ✅ 「Edit Type & Fields」ページの表示  
// ✅ Display "Edit Type & Fields" page
function smart_booking_edit_types_page() {
    $saved_data_acf = get_option('smart_booking_acf_data', array());
    $saved_data_db = get_option('smart_booking_acf_db', array());

    if (!is_array($saved_data_acf)) {
        $saved_data_acf = array();
    }
    if (!is_array($saved_data_db)) {
        $saved_data_db = array();
    }

    // ✅ 予約済み日時（reserved-daytime）を手動で追加  
    // ✅ Manually add reserved date and time (reserved-daytime)
    $reserved_daytime_data = array(
        'post_type' => 'reserved-daytime',
        'acf_group' => '予約データ',
        'acf_fields' => array(
            ['event-id', 'number'],
            ['event-name', 'text'],
            ['reserved-user-name', 'text'],
            ['reserved-user-email', 'email'],
            ['reserved-day', 'date_picker'],
            ['reserved-time', 'time_picker'],
            ['reserved-phone', 'text'],
            ['reserved-zip', 'text'],
            ['reserved-address', 'text'],
            ['reserved-building', 'text'],
            ['reserved-message', 'textarea'],
            ['reserved-auto-accept', 'true_false'],
            ['confirmation-email-enabled', 'true_false'],
            ['confirmation_template', 'select'],
            ['reminder_enabled', 'true_false'],
            ['reminder_time', 'number'],
            ['reminder_template', 'select'],
            ['reserved-age', 'text'],
            ['reserved-gender', 'text'],
            ['reserved-marital-status', 'text'],
            ['reserved-income', 'text'],
            ['reserved-residence', 'text'],
            ['reserved-rent', 'text'],
            ['reserved-people', 'text'],
            ['reserved-course', 'text'],
            ['reserved-car-count', 'text'],
            ['reserved-hobby', 'text'],
        ),
    );

    $merged_data = array_merge($saved_data_acf, $saved_data_db, [$reserved_daytime_data]);

    echo '<div class="wrap">';
    echo '<h1>Edit Type & Fields</h1>';

    if (!empty($merged_data)) {
        echo '<table class="widefat">';
        echo '<thead><tr><th>' . esc_html__('投稿タイプ', 'smart-booking-pro2') . '</th><th>' . esc_html__('ACF グループ', 'smart-booking-pro2') . '</th><th>' . esc_html__('フィールド', 'smart-booking-pro2') . '</th><th>' . esc_html__('アクション', 'smart-booking-pro2') . '</th></tr></thead>';
        echo '<tbody>';
    foreach ($merged_data as $data) {
        $post_type = $data['post_type'];
        $post_type_url = admin_url('edit.php?post_type=' . $post_type);

        echo '<tr>';

        // 投稿タイプ表示
        echo '<td>';
        echo !empty($data['post_type_label']) ? esc_html($data['post_type_label']) : esc_html($post_type);
        echo '</td>';

        // ACFグループ名
        echo '<td>';
        echo !empty($data['acf_group_label']) ? esc_html($data['acf_group_label']) : esc_html($data['acf_group']);
        echo '</td>';

        // ✅ ACFフィールド名を動的取得して表示
        echo '<td>';
        $field_names = get_acf_field_names_by_post_type($post_type);
        if (!empty($field_names)) {
            foreach ($field_names as $field_name) {
                echo esc_html($field_name) . '<br>';
            }
        } else {
            echo '<span style="color:#888;">フィールド未検出</span>';
        }
        echo '</td>';

        // 表示・削除ボタン
        echo '<td>';
        echo '<a href="' . esc_url($post_type_url) . '" class="button button-primary">' . esc_html__('表示', 'smart-booking-pro2') . '</a> ';
        echo '<form method="post" style="display:inline;" class="delete-post-type-form" data-post-type="' . esc_attr($post_type) . '">';
        echo '<input type="hidden" name="delete_post_type" value="' . esc_attr($post_type) . '">';
        echo '<input type="submit" class="button button-danger" value="' . esc_attr__('削除', 'smart-booking-pro2') . '">';
        echo '</form>';
        echo '</td>';

        echo '</tr>';
    }

        echo '</tbody></table>';
    } else {
        echo '<p>まだ作成されたデータはありません。</p>';
    }

    echo '</div>';
}


function smart_booking_cf7_to_acf() {
    check_ajax_referer('smart_booking_nonce', 'nonce');

    if (!isset($_POST['form_id']) || empty($_POST['form_id'])) {
        wp_send_json_error(['message' => esc_html__('フォームIDが指定されていません。', 'smart-booking-pro2')]);
    }

    $form_id = intval($_POST['form_id']);
    $form = get_post($form_id);

    if (!$form) {
        wp_send_json_error(['message' => esc_html__('指定されたフォームが見つかりません。', 'smart-booking-pro2')]);
    }

    // ✅ `post_content` を取得・変換
    $form_content = isset($form->post_content) ? $form->post_content : '';

    // ✅ フォームの HTML からショートコードのみを抽出（無駄な文字を除外）
    preg_match_all('/\[(\w+)([^\]]*)\]/u', $form_content, $shortcode_matches);

    $fields = [];
    if (!empty($shortcode_matches[1])) {
        foreach ($shortcode_matches[1] as $index => $shortcode_name) {
            // 無効なショートコードを除外（例: `submit` などの不要タグ）
            if (in_array($shortcode_name, ['submit', 'captcha', 'file'])) {
                continue;
            }

            // ショートコード解析
            $shortcode = "[" . $shortcode_name . $shortcode_matches[2][$index] . "]";
            $parsed_field = cf7_parse_shortcode($shortcode);

            // **有効なフィールドのみ追加**
            if (isset($parsed_field['name']) && !empty(trim($parsed_field['name']))) {
                $fields[] = $parsed_field;
            }
        }
    }

    // **空のフィールドを削除**
    $fields = array_filter($fields, function ($field) {
        return isset($field['name']) && !empty(trim($field['name']));
    });

    // **フィールドが無い場合、エラーメッセージを返す**
    if (empty($fields)) {
        wp_send_json_error(['message' => esc_html__('フォームから有効なフィールドが取得できませんでした。', 'smart-booking-pro2')]);
    }

    // **成功レスポンス**
    wp_send_json_success([
        'message' => esc_html__('フィールドの取得に成功しました。', 'smart-booking-pro2'),
        'fields'  => $fields,
    ]);

}

// ✅ AJAXアクションの登録
add_action('wp_ajax_smart_booking_cf7_to_acf', 'smart_booking_cf7_to_acf');
add_action('wp_ajax_nopriv_smart_booking_cf7_to_acf', 'smart_booking_cf7_to_acf');



// 各種リスト取得  
// Retrieve various lists

// 確認メール  
// Confirmation email
function ers_get_confirmation_templates() {
    $template_dir = plugin_dir_path(__FILE__) . '../templates/confirms';

    $txt_templates = glob($template_dir . '/confirmation-email*.txt');
    $html_templates = glob($template_dir . '/confirmation-email*.html');

    $templates = array_merge(
        is_array($txt_templates) ? $txt_templates : [],
        is_array($html_templates) ? $html_templates : []
    );

    $template_choices = [];
    foreach ($templates as $template) {
        $filename = basename($template);
        $template_choices[$filename] = $filename;
    }

    return $template_choices;
}


// リマインダー  
// Reminder
function ers_get_reminder_templates() {
    $template_dir = plugin_dir_path(__FILE__) . '../templates/reminders';
    $templates = [];

    if (is_dir($template_dir)) {
        $files = scandir($template_dir);
        foreach ($files as $file) {
            if (strpos($file, 'remainder-reservation') === 0) {
                $templates[$file] = $file;
            }
        }
    }

    return $templates;
}

// singleページテンプレート  
// Single page template
function ers_get_template_files() {
    $template_dir = plugin_dir_path(__FILE__) . '../templates/singles';
    $files = scandir($template_dir);
    $templates = array();

    foreach ($files as $file) {
        if (strpos($file, 'single-') === 0 && substr($file, -4) === '.php') {
            $templates[$file] = $file; // キーと値を同じにする
        }
    }

    return $templates;
}

// フォームテンプレート  
// Form template
function ers_get_form_templates() {
    $template_dir = plugin_dir_path(__FILE__) . '../templates/forms';
    $files = scandir($template_dir);
    $templates = array();
  
    foreach ($files as $file) {
        if (strpos($file, 'form-template') === 0 && substr($file, -4) === '.txt') {
            $templates[$file] = $file; // キーと値を同じにする
        }
    }
  
    return $templates;
  }

// cssデザイン  
// CSS design
function ers_get_formdesign_styles() {
    $css_dir = plugin_dir_path(__FILE__) . '../css/';
    $files = scandir($css_dir);
    $styles = array();

    foreach ($files as $file) {
        if (strpos($file, 'style-formdesign') === 0 && substr($file, -4) === '.css') {
            $styles[$file] = $file; // `style-formdesign-xxx.css` の一覧を取得
        }
    }

    return $styles;
}

// 投稿タイプ  
// Function to get custom post types
function get_custom_post_types_for_select() {
    $saved_acf_data = get_option('smart_booking_acf_data', array());
    $saved_acf_db = get_option('smart_booking_acf_db', array());

    $merged_data = array_merge($saved_acf_data, $saved_acf_db);

    $choices = array();

    // 「指定なし」を翻訳対応で追加
    $choices[''] = __('指定なし', 'smart-booking-pro2');

    foreach ($merged_data as $data) {
        if (isset($data['post_type'])) {
            $label = isset($data['post_type_label']) ? $data['post_type_label'] : $data['post_type'];
            $choices[$data['post_type']] = esc_html__($label, 'smart-booking-pro2');
        }
    }

    // reserved-daytime を翻訳付きで追加
    $choices['reserved-daytime'] = __('予約済み日時', 'smart-booking-pro2');

    return $choices;
}


// 管理者権限の確認  
// Check administrator permissions
function ers_set_management_no($post_id) {
    // 投稿の種類が 'event-reservation' 以外なら処理しない
    if (get_post_type($post_id) !== 'event-reservation') {
        return;
    }

    // 自動保存やリビジョンなら処理しない
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    // すでに管理番号があるなら何もしない
    if (!empty(get_post_meta($post_id, 'management-no', true))) {
        return;
    }

    // キャッシュキーを使って最大番号を取得（審査対応）
    $cache_key = 'ers_max_management_no';
    $cached_max = wp_cache_get($cache_key, 'ers');

    if ($cached_max === false) {
        global $wpdb;
        $sql = "
            SELECT MAX(CAST(meta_value AS UNSIGNED))
            FROM {$wpdb->postmeta}
            WHERE meta_key = %s
        ";
        // This query is used to retrieve the maximum value of a specific postmeta key ("management-no")
        // in order to assign a unique ID. It is securely prepared via $wpdb->prepare()
        // and is cached via wp_cache_get/set for performance.
        // The use of direct query is necessary due to the lack of a suitable WordPress core API for MAX aggregate queries.
        $cached_max = (int) $wpdb->get_var($wpdb->prepare($sql, 'management-no'));

        // キャッシュ保存（有効期限：10分）
        wp_cache_set($cache_key, $cached_max, 'ers', 600);
    }

    // 最大値に +1（または10001から）
    $new_management_no = $cached_max ? $cached_max + 1 : 10001;

    // 管理番号を保存
    update_post_meta($post_id, 'management-no', $new_management_no);

    // キャッシュ更新（この投稿で最大値が更新されるため）
    wp_cache_set($cache_key, $new_management_no, 'ers', 600);
}
add_action('save_post', 'ers_set_management_no');


function ers_update_post_slug($post_id) {
    // 自動保存時は何もしない  
    // Do nothing during autosave
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    // 投稿タイプが 'event-reservation' でない場合は何もしない  
    // Do nothing if post type is not 'event-reservation'
    if (get_post_type($post_id) !== 'event-reservation') {
        return;
    }

    // ✅ ACFの関数が存在するか確認（未インストール対策）  
    // ✅ Check if ACF functions exist (in case ACF is not installed)
    if (!function_exists('get_field')) {
        return;
    }

    // ACFフィールド 'designation-slug' の値を取得  
    // Get the value of the ACF field 'designation-slug'
    $designation_slug = get_field('designation-slug', $post_id);

    // スラッグが空の場合は何もしない  
    // Do nothing if the slug is empty
    if (empty($designation_slug)) {
        return;
    }

    // 投稿データを更新  
    // Update post data
    $post_data = [
        'ID'        => $post_id,
        'post_name' => sanitize_title($designation_slug),
    ];

    remove_action('save_post', 'ers_update_post_slug'); // 無限ループ防止
    wp_update_post($post_data);
    add_action('save_post', 'ers_update_post_slug'); // フックを再登録
}
add_action('save_post', 'ers_update_post_slug');


// ACFフィールドの管理者設定：イベント予約  
// Administrator settings for ACF fields: Event Reservation
add_filter('acf/load_field', function ($field) {
    // 管理者以外に非表示にするフィールドリスト
    $restricted_fields = [
        'calendar-display',
        'confirmation_template',
        'reminder_template',
        'designation-header-name',
        'designation-footer-name', 
        'designation-contactform7',
        'designation-thanks-page',
        'select-single-template',
        'calendar-display',
        'form-template-file',
        'formdesign-css-file',
        'datasase-db'
    ];


    return $field;
});

// スケジュールイベントを登録  
// Register scheduled event
function ers_schedule_reminder_event() {
    if (!wp_next_scheduled('ers_reminder_event')) {
        wp_schedule_event(time(), 'minute', 'ers_reminder_event'); // 毎分実行
    }
}
add_action('wp', 'ers_schedule_reminder_event');

// フックにリマインダー送信関数を登録  
// Register reminder sending function to hook
add_action('ers_reminder_event', 'ers_send_reminders');

function ers_add_cron_intervals($schedules) {
    $schedules['minute'] = [
        'interval' => 60, // 60秒（1分）
        'display'  => __('Every Minute')
    ];
    return $schedules;
}
add_filter('cron_schedules', 'ers_add_cron_intervals');

// Googleカレンダー登録関数（定型）  
// Google Calendar registration function (standard)
function generate_google_calendar_url($title, $start_datetime, $end_datetime, $details = '', $location = '') {
    // JST（Zを付けない形式）でGoogleカレンダーに渡す  
    // Pass to Google Calendar in JST format (without 'Z')
    $start = date('Ymd\THis', strtotime($start_datetime));
    $end = date('Ymd\THis', strtotime($end_datetime));

    $params = [
        'action'   => 'TEMPLATE',
        'text'     => $title,
        'dates'    => "{$start}/{$end}",
        'details'  => $details,
        'location' => $location,
    ];

    return 'https://calendar.google.com/calendar/render?' . http_build_query($params);
}

/* クローラーのインデックス排除 */  
/* Exclude from crawler indexing */
add_action('wp_head', function () {
    if (is_singular()) {
        $post_type = get_post_type();

        // sbp2-で始まる or event-reservation の投稿タイプは noindex  
        // Post types starting with sbp2- or event-reservation are set to noindex
        if (strpos($post_type, 'sbp2-') === 0 || $post_type === 'reserved-daytime') {
            echo '<meta name="robots" content="noindex, nofollow">' . "\n";
        }
    }
});

add_action('save_post', function ($post_id) {
    // 自動保存時やリビジョンは除外
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (wp_is_post_revision($post_id)) return;

    // 投稿タイプを取得
    $post_type = get_post_type($post_id);

    // sbp2系 または event-reservation のときのみ設定
    if (strpos($post_type, 'sbp2-') === 0 || $post_type === 'reserved-daytime') {
        update_post_meta($post_id, '_noindex', '1');
        update_post_meta($post_id, '_nofollow', '1');
    }
});

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// 今後の機能追加用
// For future feature additions//////////////////////////////////////////////////////////////////////////////////////////////////////
/*
//LINE Webhook
add_action('rest_api_init', function () {
    register_rest_route('linebot/v1', '/webhook', array(
        'methods'  => 'POST',
        'callback' => 'handle_line_webhook',
        'permission_callback' => '__return_true',
    ));
});

function handle_line_webhook(WP_REST_Request $request) {
    $channel = sanitize_text_field($request->get_param('channel'));
    if (!$channel) {
        error_log("❌ channel パラメータが指定されていません。");
        return new WP_REST_Response(['error' => 'Missing channel identifier'], 400);
    }

    $posts = get_posts([
        'post_type'   => 'your_post_type_slug', // ← 実際の投稿タイプ名に変更してください
        'meta_key'    => 'line_channel_identifier',
        'meta_value'  => $channel,
        'numberposts' => 1
    ]);

    if (empty($posts)) {
        error_log("❌ 該当する投稿が見つかりません: $channel");
        return new WP_REST_Response(['error' => 'Form not found'], 404);
    }

    $post_id = $posts[0]->ID;
    $access_token = get_field('line_channel_access_token', $post_id);

    $body = json_decode($request->get_body(), true);
    if (!empty($body['events'])) {
        foreach ($body['events'] as $event) {
            $type = $event['type'] ?? 'unknown';
            $userId = $event['source']['userId'] ?? 'no-user';
            error_log("[LINE Webhook][$channel][$type] UserID: $userId");

            // 🔜 ここにプロフィール取得・保存処理などを追加できます
        }
    }

    return new WP_REST_Response(['status' => 'ok'], 200);
}


add_filter('acf/load_value/name=line_webhook_url', function ($value, $post_id, $field) {
    // 識別子を取得
    $channel_key = get_field('line_channel_identifier', $post_id);

    if (!empty($channel_key)) {
        // 投稿ごとのWebhook URLを生成
        return home_url('/wp-json/linebot/v1/webhook?channel=' . urlencode($channel_key));
    }

    // 識別子が未入力なら通常のURLを表示
    return home_url('/wp-json/linebot/v1/webhook');
}, 10, 3);
*/

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Javascriptの読み込み処理///////////////////////////////////////////////////////////////////////////////////////////////////////////  
// JavaScript loading process ///////////////////////////////////////////////////////////////////////////////////////////////////////
function smart_booking_admin_scripts($hook) {
    global $pagenow;

    // 管理画面の情報を取得  
    // Get admin screen information
    $screen = get_current_screen();

    // 対象画面か判定（投稿編集画面、予約CPT画面、Smart Bookingのメニュー画面）  
    // Determine if the current screen is targeted (post edit screen, reservation CPT screen, Smart Booking menu screen)
    $should_load = false;

    if (
        in_array($pagenow, ['post.php', 'post-new.php'], true) ||
        (isset($screen->post_type) && $screen->post_type === 'event_reservation') ||
        strpos($hook, 'smart_booking_') !== false
    ) {
        $should_load = true;
    }

    if ($should_load) {
        $script_handle = 'smart-booking-admin';
        $script_path   = plugin_dir_path(__FILE__) . '../js/script-admin.js';

        wp_enqueue_script(
            $script_handle,
            plugin_dir_url(__FILE__) . '../js/script-admin.js',
            ['jquery', 'wp-i18n'],
            filemtime($script_path),
            true
        );

        // 翻訳ファイルの読み込み（languages ディレクトリ内の JSON も対象）
        wp_set_script_translations(
            $script_handle,
            'smart-booking-pro2',
            plugin_dir_path(__FILE__) . '../languages'
        );

        // AJAXやnonceの渡し
        wp_localize_script($script_handle, 'smart_booking_vars', [
            'nonce'   => wp_create_nonce('smart_booking_nonce'),
            'ajaxurl' => admin_url('admin-ajax.php'),
        ]);
    }
}
add_action('admin_enqueue_scripts', 'smart_booking_admin_scripts');

function ers_enqueue_thanks_calendar_script() {
    if (isset($_GET['calendar']) && $_GET['calendar'] === '1') {
        // JS読み込み
        wp_enqueue_script(
            'ers-thanks-calendar',
            plugins_url('../js/thanks-calendar.js', __FILE__),
            ['wp-i18n'], // ✅ 翻訳関数を使用するための依存追加
            '1.0',
            true
        );

        wp_set_script_translations(
            'ers-thanks-calendar',
            'smart-booking-pro2', // ✅ あなたのText Domainと統一
            plugin_dir_path(__FILE__) . '../languages' // ✅ .moファイルの格納ディレクトリ
        );


        // CSS読み込み
        wp_enqueue_style(
            'ers-calendar-style',
            plugins_url('../css/style-google-calendar.css', __FILE__),
            [],
            '1.0',
            'all'
        );
    }
}
add_action('wp_enqueue_scripts', 'ers_enqueue_thanks_calendar_script');

add_filter('wpcf7_form_elements', function ($form) {
    if (!is_singular()) return $form;
    $post_id = get_the_ID();
    $line_url = get_field('line_friend_url', $post_id);
    if ($line_url) {
        $form = str_replace('{line_url}', esc_url($line_url), $form ?? '');
//        $form = str_replace('{line_url}', esc_url($line_url), $form);
    }
    return $form;
});



add_action('admin_head-post.php', 'sbp2_customize_editor_ui');
add_action('admin_head-post-new.php', 'sbp2_customize_editor_ui');
add_action('admin_footer-post.php', 'sbp2_customize_draft_button_label');
add_action('admin_footer-post-new.php', 'sbp2_customize_draft_button_label');

function sbp2_customize_editor_ui() {
    global $post;

    if (!$post || !isset($post->post_type)) {
        return;
    }

    $target_types = ['reserved-daytime'];

    $merged_data = array_merge(
        get_option('smart_booking_acf_data', []),
        get_option('smart_booking_acf_db', [])
    );

    foreach ($merged_data as $data) {
        if (!empty($data['post_type'])) {
            $target_types[] = sanitize_key($data['post_type']);
        }
    }

    if (!in_array($post->post_type, $target_types, true)) {
        return;
    }

    echo '<style>
    #submitdiv .misc-pub-section.misc-pub-post-status,
    #submitdiv .misc-pub-section.misc-pub-visibility,
    #submitdiv .misc-pub-section.misc-pub-curtime,
    #submitdiv .misc-pub-section.misc-pub-revisions,
    #submitdiv #publish {
        display: none !important;
    }
s    </style>';
}


function sbp2_customize_draft_button_label() {
    global $post;

    if (!$post || !isset($post->post_type)) {
        return;
    }

    $target_types = ['reserved-daytime'];

    $merged_data = array_merge(
        get_option('smart_booking_acf_data', []),
        get_option('smart_booking_acf_db', [])
    );

    foreach ($merged_data as $data) {
        if (!empty($data['post_type'])) {
            $target_types[] = sanitize_key($data['post_type']);
        }
    }

    if (!in_array($post->post_type, $target_types, true)) {
        return;
    }

    // ✅ JS内でも翻訳された文字列を使えるようにする
    $label = esc_js(__('保存', 'smart-booking-pro2'));

    echo "<script>
    document.addEventListener('DOMContentLoaded', function () {
        const draftBtn = document.getElementById('save-post');
        if (draftBtn) {
            draftBtn.value = '{$label}';
        }
    });
    </script>";
}

//プレースホルダー用の投稿取得と生成
add_action('wp_ajax_get_acf_placeholders', function () {

    $post_type = sanitize_text_field($_POST['post_type'] ?? '');

    if (empty($post_type)) {
        wp_send_json_error(__('投稿タイプが指定されていません。', 'smart-booking-pro2'));
    }

    // ✅ ACFグループを取得（投稿がなくてもOK）
    $field_groups = acf_get_field_groups(['post_type' => $post_type]);
    $placeholders = [];

    foreach ($field_groups as $group) {
        $fields = acf_get_fields($group['key']);
        if (is_array($fields)) {
            foreach ($fields as $field) {
                if (!empty($field['name'])) {
                    $placeholders[] = [
                        'label' => $field['label'] ?? $field['name'],
                        'placeholder' => '{' . $field['name'] . '}'
                    ];
                }
            }
        }
    }

    if (empty($placeholders)) {
        wp_send_json_error(__('フィールド情報が見つかりませんでした。', 'smart-booking-pro2'));
    }

    wp_send_json_success($placeholders);
});

