<?php
// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}


function sbp2_handle_plugin_activation() {
    if (get_option('sbp2_install_id')) {
        return;
    }

    $uuid = wp_generate_uuid4();
    $user_id = get_current_user_id();
    $user = get_userdata($user_id);

    $meta = [
        'installed_at'         => current_time('mysql'),
        'admin_email'          => get_option('admin_email'),
        'installed_by_user'    => $user ? $user->user_login : __('不明', 'smart-booking-pro2'),
        'site_url'             => site_url(),
        'locale'               => get_user_locale(),
        'wp_version'           => get_bloginfo('version'),
        'php_version'          => phpversion(),
        'plugin_version'       => defined('SBP2_PLUGIN_VERSION') ? SBP2_PLUGIN_VERSION : 'unknown',
        'theme'                => wp_get_theme()->get('Name'),
        'timezone'             => date_default_timezone_get(),
        'subscription_ends_at' => '',
        'is_verified'          => false,
    ];


    $base_url = 'https://arce-experience.com';
    $max_attempts = 3;
    $attempt = 0;
    $registered = false;
    $error_reason = null;

    while ($attempt < $max_attempts && !$registered) {
        $payload = ['uuid' => $uuid] + $meta;

        $response = wp_remote_post($base_url . '/wp-json/sbp2/v1/register-install', [
            'headers' => ['Content-Type' => 'application/json'],
            'body'    => json_encode($payload),
            'timeout' => 10,
        ]);

        if (is_wp_error($response)) {
            $error_reason = __('通信エラー: ', 'smart-booking-pro2') . $response->get_error_message();
            sbp2_write_log(__('【登録エラー】', 'smart-booking-pro2') . ' ' . $error_reason);
            break;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['status']) && $body['status'] === 'ok') {
            $registered = true;

            if (!empty($body['jwt_secret'])) {
                add_option('sbp2_jwt_secret', sanitize_text_field($body['jwt_secret']));
                sbp2_write_log(__('【登録成功】JWT秘密キーを受信しました。', 'smart-booking-pro2'));
            }

            break;
        }

        if (isset($body['reason'])) {
            if ($body['reason'] === 'uuid_exists') {
                $uuid = wp_generate_uuid4();
                $attempt++;
                sbp2_write_log(__('【登録リトライ】UUID重複。再生成してリトライします。', 'smart-booking-pro2'));
                continue;
            } elseif ($body['reason'] === 'site_exists') {
                $error_reason = __('このサイトはすでに登録済みです。', 'smart-booking-pro2');
                sbp2_write_log(__('【登録拒否】', 'smart-booking-pro2') . ' ' . $error_reason);
                break;
            } else {
                $error_reason = __('認証DBエラー: ', 'smart-booking-pro2') . $body['reason'];
                sbp2_write_log(__('【登録拒否】', 'smart-booking-pro2') . ' ' . $error_reason);
                break;
            }
        } else {
            $error_reason = __('不明な応答: ', 'smart-booking-pro2') . wp_remote_retrieve_body($response);
            sbp2_write_log(__('【登録拒否】', 'smart-booking-pro2') . ' ' . $error_reason);
            break;
        }
    }


    if ($registered) {
        add_option('sbp2_install_id', $uuid);
        add_option('sbp2_install_meta', $meta);

        if (get_option('sbp2_free_db_limit') === false) {
            add_option('sbp2_free_db_limit', 2);
        }

        if (get_option('sbp2_free_form_limit') === false) {
            add_option('sbp2_free_form_limit', 2);
        }

        if (get_option('sbp2_active_plugin_count') === false) {
            add_option('sbp2_active_plugin_count', 0);
        }

        if (get_option('sbp2_active_pro_count') === false) {
            add_option('sbp2_active_pro_count', 0);
        }

        $jwt_secret = get_option('sbp2_jwt_secret');
        if ($jwt_secret) {
            $referer_url = admin_url('admin.php?id=' . urlencode($meta['installed_at']));

            $payload = [
                'uuid'    => $uuid,
                'referer' => $referer_url,
                'iat'     => time(),
                'exp'     => time() + 300,
            ];

            $base64UrlEncode = function ($data) {
                return rtrim(strtr(base64_encode(json_encode($data)), '+/', '-_'), '=');
            };

            $header = ['alg' => 'HS256', 'typ' => 'JWT'];
            $encodedHeader  = $base64UrlEncode($header);
            $encodedPayload = $base64UrlEncode($payload);
            $signature      = hash_hmac('sha256', "$encodedHeader.$encodedPayload", $jwt_secret, true);
            $encodedSignature = rtrim(strtr(base64_encode($signature), '+/', '-_'), '=');
            $jwt = "$encodedHeader.$encodedPayload.$encodedSignature";

            $verify_response = wp_remote_post($base_url . '/wp-json/sbp2/v1/complete-install', [
                'headers' => [
                    'Content-Type'  => 'application/json',
                    'Authorization' => 'Bearer ' . $jwt,
                ],
                'body' => json_encode([
                    'uuid'   => $uuid,
                    'locale' => get_user_locale(), // 🔍 追加
                ]),
                'timeout' => 10,
            ]);

            if (is_wp_error($verify_response)) {
                $msg = __('認証通信エラー: ', 'smart-booking-pro2') . $verify_response->get_error_message();
                set_transient('sbp2_install_error_reason', $msg, 60);
                sbp2_write_log(__('【認証通信エラー】', 'smart-booking-pro2') . ' ' . $msg);
                return;
            }

            $verify_body = json_decode(wp_remote_retrieve_body($verify_response), true);
            if (!isset($verify_body['status']) || $verify_body['status'] !== 'ok') {
                $message = isset($verify_body['message']) ? $verify_body['message'] : __('不明なレスポンス', 'smart-booking-pro2');
                $msg = __('認証失敗: ', 'smart-booking-pro2') . $message;
                set_transient('sbp2_install_error_reason', $msg, 60);
                sbp2_write_log(__('【認証失敗】', 'smart-booking-pro2') . ' ' . $msg);
                return;
            }

            update_option('sbp2_plugin_status', 'active');
            sbp2_write_log(__('【認証完了】プラグインステータスが active に設定されました。', 'smart-booking-pro2'));
        }
    } else {
        set_transient('sbp2_install_error_reason', $error_reason, 60);
    }

}

add_action('admin_notices', function () {

    $error = get_transient('sbp2_install_error_reason');
    if (!$error) return;

    delete_transient('sbp2_install_error_reason');
    update_option('sbp2_plugin_status', 'inactive');

    $status_page_url = admin_url('admin.php?page=smart_booking_status');

    echo '<div class="notice notice-error">';
    echo '<p><strong>' . esc_html__('❌ Smart Booking PRO2 のインストールに失敗しました。', 'smart-booking-pro2') . '</strong></p>';
    echo '<p>' . esc_html__('理由:', 'smart-booking-pro2') . ' <code>' . esc_html($error) . '</code></p>';
    echo '<p>' . esc_html__('現在の状態ではプラグインを使用できません。', 'smart-booking-pro2') . '</p>';
    echo '<p><strong>' . esc_html__('プラグインの再有効化や再インストールをお試しください。', 'smart-booking-pro2') . '</strong></p>';
    echo '<p>' . esc_html__('不明点がある場合は、ステータス管理画面からお問い合わせください。', 'smart-booking-pro2') . '</p>';
    echo '<p>';
    echo '<a href="' . esc_url($status_page_url) . '" class="button button-primary" style="margin-top:8px;">' . esc_html__('ステータス画面を開く', 'smart-booking-pro2') . '</a>';
    echo '</p>';
    echo '</div>';
});
