<?php
// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// 🔐 認証DBから送信された秘密キーを受け取って保存するエンドポイント  
// 🔐 Endpoint to receive and store the secret key sent from the authentication DB
add_action('rest_api_init', function () {
    register_rest_route('sbp2/v1', '/store-secret-key', [
        'methods'             => 'POST',
        'callback'            => 'sbp2_store_secret_key',
        'permission_callback' => '__return_true',
    ]);
});

function sbp2_store_secret_key(WP_REST_Request $request) {
    $data = $request->get_json_params();

    if (!isset($data['uuid'], $data['jwt_secret'])) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('UUID または秘密キーが不足しています', 'smart-booking-pro2'),
        ], 400);
    }

    update_option('sbp2_jwt_secret', sanitize_text_field($data['jwt_secret']));
    update_option('sbp2_install_id', sanitize_text_field($data['uuid']));

    return new WP_REST_Response([
        'success' => true,
        'message' => __('秘密キーを保存しました', 'smart-booking-pro2'),
    ], 200);
}



// 💳 決済完了通知を受け取ってステータスなど保存するエンドポイント  
// 💳 Endpoint to receive payment completion notifications and save status information
add_action('rest_api_init', function () {
    register_rest_route('sbp2/v1', '/subscription-confirmed', [
        'methods'             => 'POST',
        'callback'            => 'sbp2_store_subscription_data',
        'permission_callback' => '__return_true',
    ]);
});

function sbp2_store_subscription_data(WP_REST_Request $request) {
    $jwt = sbp2_extract_jwt($request);
    if (!$jwt || !sbp2_verify_jwt($jwt)) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('署名が無効です', 'smart-booking-pro2'),
        ], 403);
    }

    $data = $request->get_json_params();
    $uuid            = sanitize_text_field($data['uuid'] ?? '');
    $status          = sanitize_text_field($data['subscription_status'] ?? '');
    $customer_id     = sanitize_text_field($data['stripe_customer_id'] ?? '');
    $customer_email  = sanitize_email($data['stripe_customer_email'] ?? '');
    $limit_day       = sanitize_text_field($data['subscription_limit_day'] ?? '');
    $plan_type       = sanitize_text_field($data['plan_type'] ?? '');
    $payment_status  = sanitize_text_field($data['subscription_payment_status'] ?? '');

    if (!$uuid || !$status) {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('UUID またはステータスが不足しています', 'smart-booking-pro2'),
        ], 400);
    }

    update_option('sbp2_subscription_status', $status);
    update_option('sbp2_stripe_customer_id', $customer_id);
    update_option('sbp2_stripe_email', $customer_email);
    if (!empty($limit_day)) {
        update_option('sbp2_subscription_limit_day', $limit_day);
    }
    if (!empty($plan_type)) {
        update_option('sbp2_plan_type', $plan_type);
    }
    if (!empty($payment_status)) {
        update_option('sbp2_subscription_payment_status', $payment_status);
    }

    if (empty($customer_id) || empty($customer_email)) {
        if (function_exists('sbp2_check_remote_subscription_status')) {
            $result = sbp2_check_remote_subscription_status();
            if (!$result['success']) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => sprintf(
                        __('保存失敗（再取得も失敗）: %s', 'smart-booking-pro2'),
                        $result['message']
                    ),
                ], 500);
            }
        }
    }

    return new WP_REST_Response([
        'success' => true,
        'message' => __('ステータス情報を保存しました（補完済）', 'smart-booking-pro2'),
    ], 200);
}


// 🔍 JWT抽出関数  
// 🔍 JWT extraction function
function sbp2_extract_jwt(WP_REST_Request $request) {
    $auth = $request->get_header('authorization');
    if (!$auth || strpos($auth, 'Bearer ') !== 0) return false;
    return substr($auth, 7);
}

// ✅ JWT検証関数（検証用シークレットは保存済みの option を使用）  
// ✅ JWT verification function (uses the saved option as the secret for validation)
function sbp2_verify_jwt($jwt) {
    $secret = get_option('sbp2_jwt_secret');
    if (!$secret) return false;

    $parts = explode('.', $jwt);
    if (count($parts) !== 3) return false;

    list($header, $payload, $signature) = $parts;
    $expected_sig = rtrim(strtr(base64_encode(hash_hmac('sha256', "$header.$payload", $secret, true)), '+/', '-_'), '=');

    return hash_equals($expected_sig, $signature);
}
