<?php
// 🔁 プラグイン側から認証DBのサブスク状態を確認する処理  
// 🔁 Process to check subscription status from the plugin side via the authentication DB
function sbp2_check_remote_subscription_status() {
    $uuid = get_option('sbp2_install_id');
    $installed_at = get_option('sbp2_install_meta')['installed_at'] ?? '';
    $site_url = site_url();

    if (empty($uuid) || empty($installed_at)) {
        return [
            'success' => false,
            'message' => __('UUIDまたは installed_at が未設定です', 'smart-booking-pro2'),
        ];
    }

    // 🔐 referer を作成（例: 管理画面URL + ?id=installed_at）  
    // 🔐 Generate referer (e.g., admin URL + ?id=installed_at)
    $referer = admin_url('admin.php?page=smart_booking_status&id=' . urlencode($installed_at));

        $auth_base = 'https://arce-experience.com/wp-json/sbp2/v1';

    $auth_response = wp_remote_post("{$auth_base}/authenticate", [
        'method'  => 'POST',
        'timeout' => 15,
        'headers' => ['Content-Type' => 'application/json'],
        'body'    => wp_json_encode([
            'uuid'    => $uuid,
            'referer' => $referer,
            'locale'  => get_user_locale(),
        ]),
    ]);

    if (is_wp_error($auth_response)) {
        return [
            'success' => false,
            'message' => sprintf(
                __('認証通信に失敗しました: %s', 'smart-booking-pro2'),
                $auth_response->get_error_message()
            ),
        ];
    }

    $auth_body = json_decode(wp_remote_retrieve_body($auth_response), true);

    if (empty($auth_body['success']) || empty($auth_body['jwt_secret'])) {
        return [
            'success' => false,
            'message' => sprintf(
                __('JWTシークレットの取得に失敗しました: %s', 'smart-booking-pro2'),
                $auth_body['message'] ?? __('不明なエラー', 'smart-booking-pro2')
            ),
        ];
    }


    $jwt_secret = $auth_body['jwt_secret'];
    $iat = time();
    $exp = $iat + 300;
    $payload = ['uuid' => $uuid, 'iat' => $iat, 'exp' => $exp];

    $encoded_header  = base64url_encode(json_encode(['alg' => 'HS256', 'typ' => 'JWT']));
    $encoded_payload = base64url_encode(json_encode($payload));
    $signature       = base64url_encode(hash_hmac('sha256', "$encoded_header.$encoded_payload", $jwt_secret, true));
    $jwt             = "$encoded_header.$encoded_payload.$signature";


    $status_response = wp_remote_post("{$auth_base}/check-subscription-status", [
        'method'  => 'POST',
        'timeout' => 15,
        'headers' => [
            'Content-Type'  => 'application/json',
            'Authorization' => 'Bearer ' . $jwt,
        ],
        'body' => wp_json_encode([
            'uuid'    => $uuid,
            'referer' => $referer,
            'locale'  => get_user_locale(),
        ]),
    ]);

    if (is_wp_error($status_response)) {
        return [
            'success' => false,
            'message' => sprintf(
                __('状態確認に失敗しました: %s', 'smart-booking-pro2'),
                $status_response->get_error_message()
            ),
        ];
    }

    $status_body = json_decode(wp_remote_retrieve_body($status_response), true);

    if (empty($status_body['success'])) {
        return [
            'success' => false,
            'message' => sprintf(
                __('サブスクリプション状態の取得に失敗しました: %s', 'smart-booking-pro2'),
                $status_body['message'] ?? __('不明な理由', 'smart-booking-pro2')
            ),
        ];
    }


    update_option('sbp2_plugin_status', sanitize_text_field($status_body['plugin_status']));
    update_option('sbp2_subscription_status', sanitize_text_field($status_body['subscription_status']));
    update_option('sbp2_stripe_customer_id', sanitize_text_field($status_body['stripe_customer_id'] ?? ''));
    update_option('sbp2_stripe_email', sanitize_email($status_body['stripe_customer_email'] ?? ''));
    update_option('sbp2_plan_type', sanitize_text_field($status_body['plan_type'] ?? ''));
    update_option('sbp2_jwt_secret', sanitize_text_field($status_body['jwt_secret'] ?? ''));
    update_option('sbp2_subscription_payment_status', sanitize_text_field($status_body['subscription_payment_status'] ?? ''));

    if (!empty($status_body['free_limit']) && is_array($status_body['free_limit'])) {
        update_option('sbp2_free_db_limit', sanitize_text_field($status_body['free_limit']['db_limit'] ?? ''));
        update_option('sbp2_free_form_limit', sanitize_text_field($status_body['free_limit']['form_limit'] ?? ''));
    }

    if (isset($status_body['active_plugin_count'])) {
        update_option('sbp2_active_plugin_count', intval($status_body['active_plugin_count']));
    }

    if (isset($status_body['active_pro_count'])) {
        update_option('sbp2_active_pro_count', intval($status_body['active_pro_count']));
    }

    if (!empty($status_body['subscription_limit_day'])) {
        update_option('sbp2_subscription_limit_day', sanitize_text_field($status_body['subscription_limit_day']));
    }
    update_option('sbp2_last_sync_at', current_time('mysql'));

    return ['success' => true, 'data' => $status_body];
}

// JWT生成用 base64url_encode（Stripe等のAPIで使用）  
// base64url_encode for JWT generation (used in APIs like Stripe)
function base64url_encode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}
