(function($) {
  // ✅ フォールバック辞書を取得  
// ✅ Retrieve fallback dictionary
  const fallback = window['sbp2_i18n_fallback_thanks_calendar'] || {};

  // ✅ ハイブリッド翻訳関数：__() → fallback の順に参照  
// ✅ Hybrid translation function: Refer to __() then fallback
  function T__(text) {
    if (typeof wp !== 'undefined' && typeof wp.i18n?.__ === 'function') {
      const result = wp.i18n.__(text, 'smart-booking-pro2'); // ✅ 正しい呼び出し
      if (result !== text) return result;
    }
    return fallback[text]?.[1] || text;
  }


document.addEventListener('DOMContentLoaded', function () {
  const params = new URLSearchParams(window.location.search);
  const isCalendar = params.get('calendar') === '1';
  if (!isCalendar) return;

  const title = params.get('title');
  const start = params.get('start');
  const end = params.get('end');

  if (title && start && end) {
    const container = document.createElement('div');
    container.id = 'calendar-buttons-container';
    container.className = 'calendar-buttons-container';

    // --- Google カレンダー ---
    const googleBtn = document.createElement('a');
    googleBtn.href = generateGoogleCalendarUrl(title, start, end);
    googleBtn.textContent = T__('Googleカレンダー');
    googleBtn.target = '_blank';
    googleBtn.className = 'calendar-button google';

    // --- Yahoo カレンダー ---
    const yahooBtn = document.createElement('a');
    yahooBtn.href = generateYahooCalendarUrl(title, start, end);
    yahooBtn.textContent = T__('Yahoo!カレンダー');
    yahooBtn.target = '_blank';
    yahooBtn.className = 'calendar-button yahoo';

    // --- iCal（Apple/Outlook） ---
    const icsBtn = document.createElement('a');
    const icsContent = generateIcsContent(title, start, end);
    const blob = new Blob([icsContent], { type: 'text/calendar' });
    const icsUrl = URL.createObjectURL(blob);
    icsBtn.href = icsUrl;
    icsBtn.download = 'event.ics';
    icsBtn.textContent = T__('iCal形式ダウンロード');
    icsBtn.className = 'calendar-button ical';

    // ボタンをすべて追加
    container.appendChild(googleBtn);
    container.appendChild(yahooBtn);
    container.appendChild(icsBtn);
    const target = document.getElementById('calendar-link');
    if (target) {
      target.appendChild(container);
    } else {
      document.body.appendChild(container); // フォールバック（存在しない場合）
    }
  }

  function generateGoogleCalendarUrl(title, start, end) {
    const base = 'https://www.google.com/calendar/render?action=TEMPLATE';
    return `${base}&text=${encodeURIComponent(title)}&dates=${formatDate(start)}/${formatDate(end)}`;
  }

  function generateYahooCalendarUrl(title, start, end) {
    const base = 'https://calendar.yahoo.com/?v=60&view=d&type=20';
    return `${base}&title=${encodeURIComponent(title)}&st=${formatDateYahoo(start)}&et=${formatDateYahoo(end)}`;
  }

  function generateIcsContent(title, start, end) {
    return `BEGIN:VCALENDAR
VERSION:2.0
BEGIN:VEVENT
SUMMARY:${title}
DTSTART:${formatDate(start)}
DTEND:${formatDate(end)}
DESCRIPTION:${title}
END:VEVENT
END:VCALENDAR`;
  }

  function formatDate(str) {
    const date = new Date(str.replace(' ', 'T'));
    return date.toISOString().replace(/[-:]/g, '').split('.')[0] + 'Z';
  }

  function formatDateYahoo(str) {
    const date = new Date(str.replace(' ', 'T'));
    return (
      date.getFullYear().toString() +
      pad(date.getMonth() + 1) +
      pad(date.getDate()) +
      'T' +
      pad(date.getHours()) +
      pad(date.getMinutes())
    );
  }

  function pad(n) {
    return n < 10 ? '0' + n : n;
  }
});

})(jQuery);