<?php
/**
 * Plugin Name: Smart Booking PRO2
 * Description: データベースををボタン1つで作成し予約フォームと連携できるプラグイン
 * Version: 1.1
 * Author: Gee
 * Text Domain: smart-booking-pro2
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

// プラグインバージョン  
// Plugin version
define('SBP2_PLUGIN_VERSION', '1.1.1'); 



// ✅ 翻訳ファイルを読み込む（/languages フォルダ内）  
// ✅ Load translation files (from the /languages folder)
add_action('plugins_loaded', function () {
    load_plugin_textdomain('smart-booking-pro2', false, dirname(plugin_basename(__FILE__)) . '/languages');
});

// フォールバック翻訳を読み込む  
// Load fallback translations
require_once plugin_dir_path(__FILE__) . 'languages/fallback/insert_fallback_script.php';

add_action('admin_enqueue_scripts', function () {
    // 管理画面用フォールバック辞書だけ読み込む  
    // Load only the fallback dictionary for the admin screen
    sbp2_enqueue_i18n_fallback('script-admin');
    sbp2_enqueue_i18n_fallback('thanks-calendar');
    sbp2_enqueue_i18n_fallback('admin-post-editor');
    sbp2_enqueue_i18n_fallback('sbp2-status');
    sbp2_enqueue_i18n_fallback('sbp2-single-script');
    sbp2_enqueue_i18n_fallback('formbuilder');
    sbp2_enqueue_i18n_fallback('create-tags');
});

add_action('wp_enqueue_scripts', function () {
    sbp2_enqueue_i18n_fallback('sbp2-single-script');
    sbp2_enqueue_i18n_fallback('thanks-calendar');
});



// サブスクリプション判定関数  
// Subscription status check function
require_once plugin_dir_path(__FILE__) . 'assets/admin/subscription-functions.php';

// ログの記録  
// Log recording
require_once plugin_dir_path(__FILE__) . 'includes/logger.php';

// サブスクステータス管理  
// Subscription status management
require_once plugin_dir_path(__FILE__) . 'includes/subscription-status-check.php';

// 管理画面の追加  
// Add admin menu screen
require_once plugin_dir_path(__FILE__) . 'includes/admin-database.php';

// 機能の実装  
// Feature implementation
require_once plugin_dir_path(__FILE__) . 'includes/admin-functions.php';

// CF7 ⇄ ACF 変換関数の読み込み  
// Load CF7 ⇄ ACF conversion functions
require_once plugin_dir_path(__FILE__) . 'includes/acf-cf7-functions.php';

// Ajax 関数の読み込み  
// Load Ajax functions
require_once plugin_dir_path(__FILE__) . 'includes/admin-ajax.php';

// CSV出力 関数の読み込み  
// Load CSV export functions
require_once plugin_dir_path(__FILE__) . 'includes/admin-csv.php';

// カスタム投稿タイプを読み込む
require_once plugin_dir_path(__FILE__) . 'includes/custom-types.php';


//Form Builder
require_once plugin_dir_path(__FILE__) . 'includes/admin-formbuilder.php';


// ACF のフィールドを読み込む  
// Load ACF fields
require_once plugin_dir_path(__FILE__) . 'includes/acf-fields.php';

// フロント関数の読み込み  
// Load front-end functions
require_once plugin_dir_path(__FILE__) . 'includes/functions-front.php';

// インストール時のセットアップ処理を追加
require_once __DIR__ . '/includes/install-setup.php';

// Contact Form 7 の初期設定を登録  
// Register initial settings for Contact Form 7
require_once __DIR__ . '/includes/cf7-setup.php';

// 予約一覧管理画面とCSVエクスポート機能を追加  
// Add reservation list admin screen and CSV export functionality
require_once __DIR__ . '/includes/admin-reservation-list.php';

// 予約詳細の編集機能を追加  
// Add reservation detail editing functionality
require_once __DIR__ . '/includes/admin-edit-reservation.php';
require_once __DIR__ . '/includes/admin-update-reservation.php';

// リマインダー  
// Reminder

require_once __DIR__ . '/includes/reminder-emails.php';


// インストール時処理（認証情報登録）  
// Installation process (register authentication data)
require_once plugin_dir_path(__FILE__) . 'includes/functions-install.php';
register_activation_hook(__FILE__, 'sbp2_handle_plugin_activation');

// 初期インストール確認コード  
// Initial installation check code
require_once plugin_dir_path(__FILE__) . 'includes/class-tgm-plugin-activation.php';

// RESTエンドポイントの読み込み  
// Load REST endpoints
require_once plugin_dir_path(__FILE__) . 'includes/rest-endpoints.php';


add_action('admin_init', 'sbp2_enforce_plugin_status', 1);

function sbp2_enforce_plugin_status() {
    $plugin_status = get_option('sbp2_plugin_status', null);

    // ✅ ステータス未設定時（初回インストール等）はスルー  
    // ✅ Skip if status is not set (e.g., during initial installation)
    if (is_null($plugin_status)) {
        return;
    }

    // ✅ $_GET['page'] を明示的に処理（サニタイズ & アンスラッシュ）  
    // ✅ Sanitize and unslash $_GET['page'] before use
    // $_GET['page'] is read-only and comes from admin menu links.
    // Nonce verification is not necessary for this type of usage.
    $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
    $is_sbp2_page = strpos($page, 'smart_booking_') === 0;
    $is_status_page = $page === 'smart_booking_status';

    // ✅ active でない かつ ステータス画面以外 → ブロック  
    // ✅ If not active and not on the status page, block access
    if ($plugin_status !== 'active' && $is_sbp2_page && !$is_status_page) {
        $status_page_url = admin_url('admin.php?page=smart_booking_status');

        // ✅ 表示メッセージ（日本語そのまま + 国際化対応）  
        // ✅ Display message in Japanese, internationalized for translation
        $message = '
            <div style="font-family:sans-serif;text-align:center;max-width:600px;margin:2em auto;padding:2em;border:1px solid #ccc;border-radius:8px;background:#fff;">
                <h1 style="color:#c00;">' . esc_html__('Smart Booking PRO2 は現在無効です', 'smart-booking-pro2') . '</h1>
                <p>' . esc_html__('このプラグインを使用するには、有効なライセンスが必要です。', 'smart-booking-pro2') . '</p>
                <p>' . esc_html__('以下のステータス画面にアクセスして、現在の状況をご確認ください。', 'smart-booking-pro2') . '</p>
                <p>
                    <a href="' . esc_url($status_page_url) . '" style="display:inline-block;margin-top:1em;padding:0.8em 1.5em;background:#0073aa;color:#fff;text-decoration:none;border-radius:4px;">
                        ' . esc_html__('ステータス画面を開く', 'smart-booking-pro2') . '
                    </a>
                </p>
            </div>
        ';

        // @see WordPress Plugin Handbook: wp_die() is allowed for admin access control.
        wp_die($message, esc_html__('Smart Booking PRO2 - 無効状態', 'smart-booking-pro2'), ['response' => 403]);
    }
}





add_action('tgmpa_register', function () {
    $plugins = [
        [
            'name'     => __('Contact Form 7', 'smart-booking-pro2'),
            'slug'     => 'contact-form-7',
            'required' => true,
        ],
        [
            'name'     => __('Advanced Custom Fields', 'smart-booking-pro2'),
            'slug'     => 'advanced-custom-fields',
            'required' => true,
        ],
    ];

    $config = [
        'id'           => 'smart-booking-pro2',
        'menu'         => 'sbp2-install-plugins',
        'has_notices'  => true,
        'dismissable'  => false,
        'is_automatic' => true,
        'message'      => __('Smart Booking PRO2 をご利用いただくには、以下のプラグインのインストール・有効化が必要です。', 'smart-booking-pro2'),
    ];

    tgmpa($plugins, $config);
});



register_activation_hook(__FILE__, function () {
    add_option('ers_needs_initial_cf7_form', 1);
    add_option('ers_needs_initial_sample_post', 1);
});

function smart_booking_enqueue_admin_styles($hook) {
    // `smart_booking_create_cf7` のページだけ適用  
    // Apply only to the `smart_booking_create_cf7` page
    if ($hook !== 'smart_booking_page_smart_booking_create_cf7') {
        return;
    }

    wp_enqueue_style(
        'smart-booking-create-form-css',
        plugin_dir_url(__FILE__) . 'css/style-create-form.css',
        array(),
        filemtime(plugin_dir_path(__FILE__) . 'css/style-create-form.css')
    );
}
add_action('admin_enqueue_scripts', 'smart_booking_enqueue_admin_styles');



require_once plugin_dir_path(__FILE__) . 'includes/permissions.php';


register_activation_hook(__FILE__, 'ers_add_custom_capabilities');
register_deactivation_hook(__FILE__, 'ers_remove_custom_capabilities');


add_action('admin_enqueue_scripts', function () {
    echo '<style>
        .ers-menu-badge {
            background: #ff4081;
            color: white;
            font-size: 9px;
            font-weight: bold;
            padding: 2px 5px;
            border-radius: 3px;
            margin-left: 5px;
            vertical-align: middle;
            display: inline-block;
        }
    </style>';
});

function sbp2_enqueue_datepicker_assets() {
    // jQuery UI本体のdatepickerを確実に読み込む  
    // Ensure the jQuery UI core datepicker is loaded
    wp_enqueue_script('jquery-ui-datepicker');

    // 必要ならtimepickerアドオンも読み込む（すでに読み込まれてるなら不要）  
    // Load timepicker addon if needed (skip if already loaded)
    wp_enqueue_script(
        'jquery-ui-timepicker-addon',
        plugin_dir_url(__FILE__) . 'js/jquery-ui-timepicker-addon.min.js',
        ['jquery', 'jquery-ui-datepicker'],
        '1.6.3',
        true
    );

    // jQuery UIのCSS（WordPress本体にはないのでCDN等から）  
    // jQuery UI CSS (not included in WordPress core, so load from CDN or similar)
    wp_enqueue_style(
        'jquery-ui-css',
        plugin_dir_url(__FILE__) . 'assets/css/jquery-ui.css',
        [],
        '1.13.2'
    );

}
add_action('wp_enqueue_scripts', 'sbp2_enqueue_datepicker_assets');


add_action('admin_enqueue_scripts', 'sbp2_enqueue_admin_assets');
add_action('admin_enqueue_scripts', 'sbp2_enqueue_admin_assets');
function sbp2_enqueue_admin_assets($hook) {

    // 共通スクリプト（CryptoJSなど）  
    // Common scripts (e.g., CryptoJS)
    wp_enqueue_script(
        'crypto-js',
        plugin_dir_url(__FILE__) . 'assets/js/crypto-js.min.js',
        [],
        '4.1.1',
        true
    );

    // タブ記録リセットスクリプトは、全ページで読み込み  
    // Load the tab memory reset script on all pages
    wp_enqueue_script(
        'sbp2-clear-tab',
        plugin_dir_url(__FILE__) . 'js/sbp2-clear-localstrage.js',
        [],
        '1.0',
        true
    );

    /// 投稿タイプ判定のために画面情報取得（全hook共通）  
    // Get screen information to determine post type (common to all hooks)

    $screen = get_current_screen();
    if (!$screen || empty($screen->post_type)) return;

    // 許可された投稿タイプ一覧  
    // List of allowed post types
    $allowed_post_types = ['reserved-daytime'];
    $saved_data_acf = get_option('smart_booking_acf_data', []);
    $saved_data_db  = get_option('smart_booking_acf_db', []);
    $merged_data = array_merge(
        is_array($saved_data_acf) ? $saved_data_acf : [],
        is_array($saved_data_db)  ? $saved_data_db  : []
    );
    foreach ($merged_data as $data) {
        if (!empty($data['post_type'])) {
            $allowed_post_types[] = $data['post_type'];
        }
    }

    // 投稿編集画面：admin-post-editor.js を読み込み  
    // Load admin-post-editor.js on the post editing screen
    if (in_array($hook, ['post-new.php', 'post.php'], true)) {
        if (in_array($screen->post_type, $allowed_post_types, true)) {
            wp_enqueue_script(
                'custom-admin-post-editor',
                plugin_dir_url(__FILE__) . 'js/admin-post-editor.js',
                ['wp-i18n'], // ✅ 翻訳依存追加
                '1.1',
                true
            );

            wp_set_script_translations(
                'custom-admin-post-editor',
                'smart-booking-pro2',
                plugin_dir_path(__FILE__) . 'languages'
            );
        }
    }

    // 投稿一覧画面：admin-post-list-editor.js を読み込み  
    // Load admin-post-list-editor.js on the post list screen
    if ($hook === 'edit.php') {
        if (in_array($screen->post_type, $allowed_post_types, true)) {
            wp_enqueue_script(
                'custom-admin-post-list-editor',
                plugin_dir_url(__FILE__) . 'js/admin-post-list-editor.js',
                [],
                '1.0',
                true
            );
        }
    }
}

// thanksページを登録  
// Register the thanks page
register_activation_hook(__FILE__, 'sbp2_create_thanks_page');

// フック登録は add_filter を main に書いても OK、または install-setup.php 内で呼んでもOK  
// You can register the hook with add_filter in main, or call it inside install-setup.php
add_filter('theme_page_templates', 'sbp2_register_page_template');
add_filter('template_include', 'sbp2_load_custom_page_template');


// ✅ WordPress プラグイン側: PHP (smart-booking-pro2.php 内)  
// ✅ WordPress plugin side: PHP (inside smart-booking-pro2.php)
add_action('admin_enqueue_scripts', 'sbp2_enqueue_admin_scripts');
function sbp2_enqueue_admin_scripts($hook_suffix) {


    wp_enqueue_script(
        'sbp2-status-js',
        plugin_dir_url(__FILE__) . 'assets/js/sbp2-status.js',
        ['jquery', 'wp-i18n'], // ✅ 翻訳機能を使うなら追加
        '1.0',
        true
    );
    wp_set_script_translations(
        'sbp2-status-js',
        'smart-booking-pro2',
        plugin_dir_path(__FILE__) . 'languages'
    );



    // UUID発行時にメタとして保存した情報も取得  
    // Also retrieve metadata saved at the time of UUID generation
    $meta = get_option('sbp2_install_meta');
    $installed_at = isset($meta['installed_at']) ? $meta['installed_at'] : '';
    $referer_url = admin_url('admin.php?id=' . urlencode($installed_at));

    wp_localize_script('sbp2-status-js', 'SBP2', [
        'ajaxurl'             => admin_url('admin-ajax.php'),
        'uuid'                => get_option('sbp2_install_id'),
        'site_url'            => get_site_url(),
        'admin_email'         => get_option('admin_email'),
        'wp_version'          => get_bloginfo('version'),
        'php_version'         => phpversion(),
        'plugin_version'      => SBP2_PLUGIN_VERSION,
        'installed_at'        => $installed_at,
        'installed_by_email'  => isset($meta['installed_by_email']) ? $meta['installed_by_email'] : '',
        'installed_by_user'   => isset($meta['installed_by_user']) ? $meta['installed_by_user'] : '',
        'theme'               => wp_get_theme()->get('Name'),
        'referer'             => $referer_url,
        'auth_url'            => 'https://arce-experience.com/wp-json/sbp2/v1/create-jwt',
        'portal_url'          => 'https://arce-experience.com/wp-json/sbp2/v1/create-customer-portal',
    ]);
}

// サブスクリプションのステータス確認処理  
// Subscription status verification process
add_action('wp_login', function ($user_login) {

        sbp2_check_remote_subscription_status();
});


register_activation_hook(__FILE__, function () {
    if (!wp_next_scheduled('sbp2_daily_subscription_check')) {
        wp_schedule_event(strtotime('03:00:00'), 'daily', 'sbp2_daily_subscription_check');
    }
});

add_action('sbp2_daily_subscription_check', 'sbp2_check_remote_subscription_status');

register_deactivation_hook(__FILE__, function () {
    wp_clear_scheduled_hook('sbp2_daily_subscription_check');
});

// xmlrpc.php を無効化  
// Disable xmlrpc.php
add_filter('xmlrpc_enabled', '__return_false');


// シングルページのJavaScript登録とローカライズ  
// Register and localize JavaScript for single pages
add_action('wp_enqueue_scripts', function () {
    if (!is_singular('event-reservation')) return;

    // 🔐 ACFが存在しない場合はスキップ  
    // 🔐 Skip if ACF is not available
    if (!function_exists('get_field')) return;

    $event_id = get_the_ID();
    if (!$event_id || !is_numeric($event_id)) return;

    wp_enqueue_script(
        'sbp2-single-script',
        plugin_dir_url(__FILE__) . 'assets/js/sbp2-single-script.js',
        ['jquery', 'wp-i18n'], // ✅ 翻訳に必要
        filemtime(plugin_dir_path(__FILE__) . 'assets/js/sbp2-single-script.js'),
        true
    );

    wp_set_script_translations(
        'sbp2-single-script',         // スクリプトハンドル
        'smart-booking-pro2',         // text domain（.potで使ったものと一致）
        plugin_dir_path(__FILE__) . 'languages' // 翻訳ファイルの格納先
    );


    // 🔐 安全な get_field ラッパー（null許容 + 型保証）  
    // 🔐 Safe get_field wrapper (null-tolerant with type guarantee)
    $acf_get = fn($key) => function_exists('get_field') ? get_field($key, $event_id) : null;

    $event_data = [
        'post_id'                         => $event_id,
        'event_start_date'               => $acf_get('event-days-start') ?? '',
        'event_end_date'                 => $acf_get('event-days-end') ?? '',
        'event_start_time'               => $acf_get('start-time') ?? '',
        'event_end_time'                 => $acf_get('end-time') ?? '',
        'event_interval_time'            => $acf_get('interval-time') ?? 30,
        'event_rest_days'                => $acf_get('rest-days') ?: [],
        'cf7_form_id'                    => $acf_get('designation-contactform7') ?? '',
        'thanks_page_slug'               => $acf_get('designation-thanks-page') ?? '',
        'event_cutoff_time'              => $acf_get('cutoff-times') ?? 0,
        'event_management_no'            => $acf_get('management-no') ?? '',
        'event_auto_accept'              => $acf_get('auto-accept') ? 1 : 0,
        'event_max_day_reserved'         => $acf_get('max-day-reserved') ?? 100,
        'event_max_time_reserved'        => $acf_get('max-time-reserved') ?? 30,
        'calendar_display'               => $acf_get('calendar-display') ? 1 : 0,
        'event_stop_reservation_days'    => array_map('trim', explode(',', $acf_get('stop-reservation-days') ?? '')),
        'event_reminder_enabled'         => $acf_get('reminder_enabled') ? 1 : 0,
        'event_reminder_time'            => $acf_get('reminder_time') ?? 60,
        'event_reminder_template'        => $acf_get('reminder_template') ?? 'remainder-reservation-default.txt',
        'event_confirmation_email_enabled' => $acf_get('send-confirmation-email') ? 1 : 0,
        'event_confirmation_template'    => $acf_get('confirmation_template') ?? 'confirmation-email-default.txt',
        'event_calendaradd_enabled'      => $acf_get('calendaradd_enabled') ? 1 : 0,
        'database_db'                    => $acf_get('datasase-db') ?? '',
        'form_template'                  => $acf_get('form-template-file') ?? '',
        'ers_nonce'                      => wp_create_nonce('ers_reservation_nonce'),
        'custom_header'                 => $acf_get('designation-header-name') ?? '',
        'custom_footer'                 => $acf_get('designation-footer-name') ?? '',
    ];

    wp_localize_script('sbp2-single-script', 'sbp2Vars', $event_data);
});

//thanksページのリプレイス
add_filter('template_include', function ($template) {
    if (is_page('sample-thanks')) {
        $custom_template = plugin_dir_path(__FILE__) . 'templates/thanks/page-thanks.php';
        if (file_exists($custom_template)) {
            return $custom_template;
        }
    }
    return $template;
});